/*******

MICE server

copyright 2001, 2002, Alan H. Clifford.

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your option)
any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc., 59
Temple Place, Suite 330, Boston, MA 02111-1307 USA


Alan Clifford can be contacted at mice@clifford.ac

Latest version at http://www.clifford.ac

$Id: mice.server.c,v 4.106 2002/07/27 22:19:47 alan Exp $
$Name: release4_2 $

 *******/


#include "mice.server.h"
#include "mice.codes.h"


#define CLIENTBUFFSIZE 512
#define SENDBUFFSIZE 512
#define MYNAMESIZE 131  // should be enough
#define NUMBERBUFFSIZE 30  // for char array to hold a number

// server type
#define SERVER_UNDEFINED 0
#define SERVER_STANDALONE 1
#define SERVER_DAEMON 2
#define SERVER_INETD 3
// server mode
// also uses SERVER_UNDEFINED
#define SERVER_MICE 1
#define SERVER_DEMO 2

// Real server
#ifndef MICESCRIPTDIR
#define MICESCRIPTDIR "/usr/local/mice/"  // must have trailing /
#endif
#define MICEPORT 5022   // port we're listening on
#define MICECONNECTPROGRAM "mice.connect"
#define MICEINFOPROGRAM "mice.info"
#define MICEKILLCONNECTPROGRAM "mice.killconnect"
#define MICEKILLINFOPROGRAM "mice.killinfo"
#define MICESELECTTIMEOUTSEC 30
#define MICELOGIDENT "mice"
#define MICEINETDIDENT "mice"
// data
#ifndef MICEDATADIR
#define MICEDATADIR "/usr/local/mice/" // must have trailing /
#endif
#define MICECONFFILE "mice.conf"

// Demo server
#ifndef DEMOSCRIPTDIR
#define DEMOSCRIPTDIR "./" // must have trailing /
#endif
#define DEMOPORT 1953   // port we're listening on
#define DEMOCONNECTPROGRAM "mice.demo.connect"
#define DEMOINFOPROGRAM "mice.demo.info"
#define DEMOKILLCONNECTPROGRAM "mice.demo.killconnect"
#define DEMOKILLINFOPROGRAM "mice.demo.killinfo"
#define DEMOSELECTTIMEOUTSEC 30
#define DEMOLOGIDENT "mice demo"
#define DEMOINETDIDENT "mice.demo"
// data
#ifndef DEMODATADIR
#define DEMODATADIR "./" // must have trailing /
#endif
#define DEMOCONFFILE "mice.demo.conf"

// client types (for AYT)
#define CLIENTUNKNOWN 0
#define CLIENTCONSOLE 1
#define CLIENTJAVA    2
#define CLIENTTELNET  3



PARAMS2 * Params2p = NULL;  // real or demo server



char *Mice_server_version = "Mundungus Internet Connection Enhancer server\r\n($Revision: 4.106 $)\r\n($Name: release4_2 $)";
char *Copyright = "Copyright 2001, 2002, Alan H. Clifford";
char *Disclaimer = "MICE server comes with ABSOLUTELY NO WARRANTY";

typedef struct {
    char * port_p;
    unsigned short int port_hs;
    char * conffile_p;
    int server_type;  // These are defines.  0 = undefined, 1 = standalone, 2 = daemon, 3 = inetd
    int server_mode;  // see defines
    int V_flag; // used to exit from main if -V option
} STARTUP_ARGS; // ready for more options



/* structure for holding each client's data */
typedef union {
    unsigned both:2;
    struct {
        unsigned connected:1;   // records the connection request
        unsigned reconnected:1; // flag that connection request should be kept
    } flag;
} CONNECTED_FLAGS;

typedef struct {
    int fd;
    char buff[CLIENTBUFFSIZE]; /*must be an array for sizeof */
    int buffsize;
    char *buffptr;  /* receive to here */
    char *lineptr;  /* remember the start of new line.  Needed if two commands in one recv */
    CONNECTED_FLAGS connected;
    char *nameptr;
    char *ip_addressptr;
    char remoteid[NUMBERBUFFSIZE]; // client's pid on remote machine
    char remoteowner[NUMBERBUFFSIZE]; // client's euid on remote machine
    /* send buffer */
    char * sendbuff;  /* dynamically allocated */
    int sendbuffsize;
    int bytes_sent, bytes_notsent;
    short int donotsend; /* set to 1 if client is not to receive data */
    short int has_id; /* set to 1 if client sends server id */
    int client_type; // 0 = unknown, 1 = linux console, 2 = java, 3 = telnet
    int ayt_count;  // initialized to 0
} CLIENT_DATA;

pid_t Child_pid; /* will be set to parent - for testing as it can't be this pid */
pid_t Info_oneshot_pid; /* will be set to parent - for testing as it can't be this pid */
// int Child_pid_status;
int Connect_requests = 0;
int Reconnect_requests = 0;
/* Connect_requests is incremented for each connect request, decremented for
 each disconnect request and reset to zero in zombie handler.
 Reconnect_requests are not touched in the zombie handler */
int Child_status_flags = 0;
/* bit 0 and 1 set in parent after connector child fork
 bit 1 reset to 0 in zombie handler
 bit 0 reset to 0 after loss of connector child recognized
 */
// int Online_flag = 0; // 1 = online, -1 = offline, 0 = not valid


// void setparams(int real_or_demo);

int create_socket(STARTUP_ARGS *startup);
char *getservername(void);
void sigchld_handler(int s);
CLIENT_DATA * get_client_data(int fd);
CLIENT_DATA * create_client_data(int fd);
CLIENT_DATA * remove_client_data(int fd);
int is_client_connected(int fd);
int is_client_reconnected(int fd);
int is_client_connected_either(int fd);
int is_client_donotsend(int fd);
void set_client_owner(int fd, char *owner_p);
void set_client_id(int fd, char *id_p);
int kill_another_client(int calling_fd, char *tokill_id_p);
void set_client_connected(int fd);
void set_client_reconnected(int fd);
void set_client_connected_both(int fd);
void set_client_unconnected(int fd);
void set_client_unconnected_both(int fd);
void set_client_unreconnected(int fd);
void set_all_clients_unconnected(void);
void set_all_clients_unreconnected(void);
void set_client_donotsend(int fd);
int get_ayt_status(int fd, int *type, int *count);
void increment_ayt_count(int fd);
int reset_ayt_count(int fd, int type);
char *get_client_name(int fd);
char *get_client_address(int fd);
int get_client_count(void);
int set_server_id_flag(int fd, char * test_id);
int is_server_id_ok(int fd);
void * client_data_set(int action, int fd);
void * client_data_set2(int action, int fd, char *text);
CLIENT_DATA * alloc_client_data(void);
CLIENT_DATA * free_client_data(CLIENT_DATA *client_data_ptr);
void more_client_data(int fd, CLIENT_DATA *data_ptr);
void s_queue_ayt(int client_flag, int client_fd, int listener,
                 int fdmax, fd_set *masterptr, fd_set *writemasterptr);
void s_queue_ayt_2(int client_fd, int listener, int fdmax,
                   fd_set *masterptr, fd_set *writemasterptr);
void s_queue_cstring(char *s, int client_flag, int client_fd, int listener,
                     int fdmax, fd_set *masterptr, fd_set *writemasterptr);
void s_queue_nbytes(char *s, int nbytes, int client_flag, int client_fd, int listener,
                    int fdmax, fd_set *masterptr, fd_set *writemasterptr, int sendoverride);
char *s_queue_alloc(CLIENT_DATA * client_data_ptr, int nbytes);
void s_queue_send(int client_fd, int listener, fd_set *masterptr, fd_set *writemasterptr);
void s_queue_status(int client_flag, int client_fd, int listener, int fdmax,
                    fd_set *masterptr, fd_set *writemasterptr);  /* send status */
pid_t s_queue_do_info_oneshot(int client_flag, int client_fd, int listener, int fdmax,
                              fd_set *masterptr, fd_set *writemasterptr); /* child cannot queue sends */
pid_t s_queue_do_connect(int client_flag, int client_fd, int listener, int fdmax,
                         fd_set *masterptr, fd_set *writemasterptr); /* child cannot queue sends */
int check_for_runaway(void);
void check_for_client_newline(int nbytes, CLIENT_DATA *client_data_ptr, int client_fd,
                          int listener, int fdmax, fd_set *masterptr, fd_set *writemasterptr);
int check_and_do_command(char *client_string, int client_fd,
                          int listener, int fdmax, fd_set *masterptr, fd_set *writemasterptr);
void s_queue_line_status(int client_flag, int client_fd, int listener, int fdmax,
                         fd_set *masterptr, fd_set *writemasterptr);
void conditional_disconnect(void);
void do_disconnect(pid_t pid_to_kill, char *exec_p, char const  *program_p);
int check_telnet(char *linep, int len_data);
char *getnow_s(void);
int is_reconnect_required(int listener, int fdmax, fd_set *masterptr);
void auto_reconnect(int listener, int fdmax, fd_set *masterptr, fd_set *writemasterptr);
void kill_donotsend_clients(int listener, int fdmax, fd_set *masterptr, fd_set *writemasterptr);
char * message_data_set(char *message, int action);
char *get_message(void);
void set_message(char *message);
int cmp_message(void);
int get_options(int argc, char *argv[], STARTUP_ARGS *startup);
void do_help(void);
char *get_magic_id(void);
int get_online_flag(void);
void set_online_flag(int flag);
int cmp_online_flag(void);
int online_flag_set(int flag, int action);

PARAMS2 * paramsport(unsigned short int newport);
PARAMS2 * paramsservermode(int mice_or_demo);
PARAMS2 * paramsservertype(int newservertype);
PARAMS2 * paramsupdate(int mice_or_demo, int newservertype, char *newscript,
                       char *newconf, unsigned short int newport);
char * build_ufn(char * ufn, const char *dir, const char * file);
void showparams(void);
unsigned short int check_port(char *port_p);



PARAMS2 * paramsport(unsigned short int newport)
    // update the port
{
    return paramsupdate(SERVER_UNDEFINED, SERVER_UNDEFINED, NULL, NULL, newport);
}

PARAMS2 * paramsservermode(int mice_or_demo)
    // update the server mode
{
    return paramsupdate(mice_or_demo, SERVER_UNDEFINED, NULL, NULL, 0);
}

PARAMS2 * paramsservertype(int newservertype)
// update the server type
{
    return paramsupdate(SERVER_UNDEFINED, newservertype, NULL, NULL, 0);
}

PARAMS2 * paramsupdate(int mice_or_demo, int newservertype, char *newscript,
                       char *newconf, unsigned short int newport)
{
    // mice_or_demo, 1 = real, 2 = demo, ignore if 0 or anything else
    // server_type, see defines for inetd, daemon etc
    // newscript directory ignore if NULL
    // newdata directory ignore if NULL
    // newport, ignored if 0
    static PARAMS2 params = {
        0,
        SERVER_UNDEFINED,
        NULL, /* script dir */
        MICECONNECTPROGRAM, MICEINFOPROGRAM,
        MICEKILLCONNECTPROGRAM, MICEKILLINFOPROGRAM,
        MICELOGIDENT, MICEINETDIDENT,
        NULL,  /* connect ufn */
        NULL, /* info ufn */
        NULL, /* kill connect ufn */
        NULL, /* kill info ufn */
        NULL /* data dir */,
        MICECONFFILE,
        NULL, /* conf ufn */
        {MICESELECTTIMEOUTSEC, 0}
    };
    // directories could be alloced or a define, so need a new pointer
    // Cannot use the params. pointers as realloc is used.
    static char * scriptdir = NULL;
    static char * conf = NULL;
    static short unsigned int port = 0;
    static int servertype = SERVER_UNDEFINED;
    char *ptr;
    // update the global
    Params2p = &params;
    // command line parameters for demo and data directory can come in any order
    // and the script directory can come from the data file
    if (SERVER_MICE == mice_or_demo) { // real
        params.connectprogram = MICECONNECTPROGRAM;
        params.infoprogram = MICEINFOPROGRAM;
        params.killconnectprogram = MICEKILLCONNECTPROGRAM;
        params.killinfoprogram = MICEKILLINFOPROGRAM;
        params.logident = MICELOGIDENT;
        params.inetdident = MICEINETDIDENT;
        params.conffile = MICECONFFILE;
        params.selecttimeout.tv_sec = MICESELECTTIMEOUTSEC;
        params.selecttimeout.tv_usec = 0;
    }
    else if (SERVER_DEMO == mice_or_demo) { // demo
        params.connectprogram = DEMOCONNECTPROGRAM;
        params.infoprogram = DEMOINFOPROGRAM;
        params.killconnectprogram = DEMOKILLCONNECTPROGRAM;
        params.killinfoprogram = DEMOKILLINFOPROGRAM;
        params.logident = DEMOLOGIDENT;
        params.inetdident = DEMOINETDIDENT;
        params.conffile = DEMOCONFFILE;
        params.selecttimeout.tv_sec = DEMOSELECTTIMEOUTSEC;
        params.selecttimeout.tv_usec = 0;
    }
    // otherwise keep the same, default is the real server

        //  port
    if (0 != newport) port = newport;
    // otherwise leave alone

    if (SERVER_UNDEFINED != newservertype) servertype = newservertype;
    // check later to see if it is an OK type
    // otherwise keep the same

    if (NULL != newscript) {
        // new script directory
        // newscript should not be passed to here unless it has been checked
        // that it exists, is a directory and is accessible
        // but check again
        struct stat statbuff;
        if ( (0 == stat(newscript, &statbuff))
             && ((S_ISDIR(statbuff.st_mode)))
             && (0 == access(newscript, X_OK)) ) {
            if (NULL == (ptr = realloc(scriptdir, strlen(newscript) + 2))) { // allow for null byte and /
                // error
                syslog(LOG_USER | LOG_INFO, "script dir realloc error");
                exit(1);
            }
            else {
                scriptdir = ptr;
            strcpy(scriptdir, newscript);
            // append / if neccessary
            if (('\0' == *scriptdir) || ('/' != *(scriptdir + strlen(scriptdir) - 1)))
                strcat(scriptdir, "/");
            }
        }
        else {
            perror(newscript);
        }
    }

    if (NULL != newconf) {
        // new conf file
        // newconf should not be passed to here unless it has been checked
        // that it exists, is a regular file and is readable
        // but check again

        struct stat statbuff;
        if ( (0 == stat(newconf, &statbuff))
            && ((S_ISREG(statbuff.st_mode)))
            && (0 == access(newconf, R_OK)) ) {
            if (NULL == (ptr = realloc(conf, strlen(newconf) + 1))) { // allow for null byte and /
                // error
                syslog(LOG_USER | LOG_INFO, "data dir realloc error");
                exit(1);
            }
            else {
                conf = ptr;
                strcpy(conf, newconf);
            }
        }
        else {
            perror(newconf);
        }
    }


    // always reconstruct the structure

    {
        struct servent *miceent;
        if (0 != port) params.port = port;
        else if (NULL != (miceent = getservbyname(params.inetdident, "tcp"))) {
            // look for mice or mice.demo
            params.port = ntohs(miceent->s_port);
        }
        else if (DEMOLOGIDENT == params.logident) params.port = DEMOPORT;
        else params.port = MICEPORT;
    }

    // server type
    if ((SERVER_STANDALONE == servertype)
        || (SERVER_DAEMON == servertype)
        || (SERVER_INETD == servertype))
        params.servertype = servertype;
    else params.servertype = SERVER_DAEMON;

    // do the directories
    // put the script dir in the structure
    // use the defines if no directory has ever been specified
    if (NULL != scriptdir) params.scriptdir = scriptdir;
    else if (DEMOLOGIDENT== params.logident) params.scriptdir = DEMOSCRIPTDIR; // demo server
    else params.scriptdir = MICESCRIPTDIR; // real server

    // put the data file in the structure
    // otherwise use the defines for directory and recosntruct ufn
    if (NULL != conf) params.conf = conf;
    else if (DEMOLOGIDENT== params.logident) params.datadir = DEMODATADIR; // demo server
    else params.datadir = MICEDATADIR; // real server


    // alway reconstruct the ufns
    params.execconnect = build_ufn(params.execconnect, params.scriptdir,
                                   params.connectprogram);
    params.execinfo = build_ufn(params.execinfo, params.scriptdir,
                                   params.infoprogram);
    // params.execkill = build_ufn(params.execkill, params.scriptdir,
    //                                params.killprogram);
    params.execkillconnect = build_ufn(params.execkillconnect, params.scriptdir,
                                       params.killconnectprogram);
    params.execkillinfo = build_ufn(params.execkillinfo, params.scriptdir,
                                    params.killinfoprogram);
    if (NULL == conf)  // no command line data file
        params.conf = build_ufn(params.conf, params.datadir,
                                params.conffile);

    // lets see it
    // showparams();


    return &params;
}

char * build_ufn(char * ufn, const char *dir, const char * file)
{
    // ufn must point to malloced memory or be NULL
    // could return NULL
    char *ptr;
    if (NULL == (ptr = realloc(ufn, strlen(dir) + strlen(file) + 1))) {
        // error
        syslog(LOG_USER | LOG_INFO, "ufn realloc error");
        exit(1);
    }
    else {
        strcpy(ptr, dir);
        strcat(ptr, file);
    }
    return ptr;
}

void showparams(void)
{
    char *servertype = NULL;
    // server type
    servertype = (SERVER_STANDALONE == Params2p->servertype) ? "verbose console"
        : (SERVER_DAEMON == Params2p->servertype) ? "daemon"
        : (SERVER_INETD == Params2p->servertype) ? "inetd"
        : "unknown";

    // printf("%20s%hu\n%20s%d\n", "Port:", Params2p->port, "Server type:", Params2p->servertype);
    printf("%20s%hu\n%20s%s\n", "Port:  ", Params2p->port, "Server type:  ", servertype);
    printf("%20s%s\n%20s%s\n", "Log ident:  ", Params2p->logident,
           "Inetd service:  ", Params2p->inetdident);
    printf("%20s%s\n%20s%s\n%20s%s\n%20s%s\n%20s%s\n\n",
           "Scripts:  ", Params2p->execconnect,
           " ", Params2p->execinfo,
           " ", Params2p->execkillconnect,
           " ", Params2p->execkillinfo,
           "Data file:  ",  Params2p->conf);
}


int main(int argc, char *argv[]) /*FOLD00*/
{
    fd_set master;   // master file descriptor list
    fd_set read_fds; // temp file descriptor list for select()
    fd_set writemaster;
    fd_set write_fds;
    struct sockaddr_in remoteaddr; // client address
    int fdmax;        // maximum file descriptor number
    int listener;     // listening socket descriptor
    int newfd;        // newly accept()ed socket descriptor
    int nbytes;

    int addrlen;
    int fd_i;
    struct timeval select_tv;
    int select_rc;
    time_t force_select_t = 0; /* used to force select time out for status sending */
    long int delta_select;
    CLIENT_DATA * client_data_ptr;
    struct sigaction sa;  /* for child zombie */
    char data_to_send[CLIENTBUFFSIZE]; /* buffer to send data to clients */
    int select_timeout_sec = 30; // will be changed
    STARTUP_ARGS startup;
    char * myname_p = NULL;


    // open the log for setparams() to use then close it so that the proper
    // log ident can be used later
    openlog("Mice server", LOG_NDELAY, LOG_USER); // start logging

    get_options(argc, argv, &startup);


    // update params with startup arguments
    // at least one paramsupdate is required
    paramsupdate(startup.server_mode, startup.server_type, NULL,
                       startup.conffile_p, startup.port_hs);
    closelog();
    openlog(Params2p->logident, 0, LOG_USER); // start logging
    // read the conf file for a new script directory and update params
    {
        char *ptr;
        if (NULL != (ptr = readscriptdir(Params2p->conf)))
            paramsupdate(SERVER_UNDEFINED, SERVER_UNDEFINED, ptr, NULL, 0);
    }

    // change directory
    if (0 != chdir(Params2p->scriptdir)) {
        int errornumber = errno;
        // perror("chdir");
        syslog(LOG_USER | LOG_WARNING, "chdir to %s failed: %s",
               Params2p->scriptdir, strerror(errornumber));
        // if (SERVER_STANDALONE == Params2p->servertype) exit(7);
    }

    getisps();
    getayt();  // initialize

    if ((0 != startup.V_flag) || (SERVER_STANDALONE == startup.server_type)) {
        // lets see it
        char pathbuffer[PATH_MAX];
        int ayt_threshold;

        showparams();
        if (NULL == getcwd(pathbuffer, PATH_MAX))
            perror("error finding current directory");
        else printf("%20s%s\n", "Current directory:  ", pathbuffer);

        ayt_threshold = getayt();
        if (0 == ayt_threshold)
            printf("%20s%s\n\n", "AYT threshold:  ", "disabled");
        else
            printf("%20s%d\n\n", "AYT threshold:  ", ayt_threshold);

        showisps();
    }
    // readscriptdir(Params2p->conf);  //temporary
    // check the scripts are executable
    if (0 != check_exec_access()) exit(1);  // logging is in function
    if (0 != startup.V_flag) exit(0);
    // exit(0);


    // testing
    // getisps();
    // getisps();
    // exit(0);


    // select_timeout_sec needs to be initalized from params
    select_timeout_sec = Params2p->selecttimeout.tv_sec;

    // inetd server?
    if (SERVER_INETD == Params2p->servertype) {
        // if (SERVER_INETD == startup.server_type) {
        switch_off_stdout();
        switch_off_stderr();
        listener = STDIN_FILENO;
    }
    else {
        if (SERVER_STANDALONE == Params2p->servertype) {
        }
        else {
            // daemonize
            switch (fork()) {
            case 0:
                // child
                switch_off_stdout();
                switch_off_stderr ();
                if (-1 == setsid()) {
                    exit(5);
                }
                break;
            case -1: // error
                exit(6);
                break;
            default:
            // in parent
                _exit(0);
                break;
            }
        }
        listener = create_socket(&startup);
        switch_off_stdin();
    }


    printf("%s\n%s\n%s\n\n", Mice_server_version, Copyright, Disclaimer);

    myname_p = getservername();



/*
    // change directory
    if (0 != chdir(Params2p->scriptdir)) {
        int errornumber = errno;
        // perror("chdir");
        syslog(LOG_USER | LOG_WARNING, "chdir to %s failed: %s",
               Params2p->scriptdir, strerror(errornumber));
        // if (SERVER_STANDALONE == Params2p->servertype) exit(7);
    }
*/


    FD_ZERO(&master);    // clear the master and temp sets
    FD_ZERO(&read_fds);
    FD_ZERO(&writemaster);
    FD_ZERO(&write_fds);


    Child_pid = getpid();  /* for testing - a child can't be this */
    Info_oneshot_pid = getpid();  /* for testing - a child can't be this */


    sa.sa_handler = sigchld_handler; // reap all dead processes
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART || SA_NOMASK;
    if (sigaction(SIGCHLD, &sa, NULL) == -1) {
        int errornumber = errno;
        syslog(LOG_USER | LOG_INFO, "Signal handler error: %s", strerror(errornumber));
        // perror("sigaction");
        // exit(1);
    }

    sa.sa_handler = sigusr1_handler;
    // Catch the usr1 signal to reload the data file
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART || SA_NOMASK;
    if (-1 == sigaction(SIGUSR1, &sa, NULL)) {
        int errornumber = errno;
        syslog(LOG_USER | LOG_INFO, "Signal handler error: %s", strerror(errornumber));
        // perror("sigaction");
        // exit(1);
    }




    // add the listener to the master set
    FD_SET(listener, &master);

    // keep track of the biggest file descriptor
    fdmax = listener; // so far, it's this one



    /* get the client data for listener - not really neccessary
     disasterous when getpeername is called!!
     client_data_ptr = client_data_set(listener, 1);
     printf("client ptr for listener = %p\n", client_data_ptr);
     */

    get_magic_id();  // initalialize to avoid it being called the first time in a fork.

    time(&force_select_t);

    // main loop
    for(;;) {
        read_fds = master; // copy it
        write_fds = writemaster; // copy it
        if (0 < (Connect_requests + Reconnect_requests)) {
            if (20L <= (delta_select = (long int)time(NULL) - (long int)force_select_t) ) {
                // if connected, force frequent timeout for status display
                select_tv.tv_sec = 0; // seconds
                select_tv.tv_usec = 1;
            }
            else {
                select_tv.tv_sec = delta_select <= 0L ? 20L : 20L - delta_select; // seconds
                select_tv.tv_usec = 1;
            }
            // printf("%ld  XXXXXXXXXXXXX\n", delta_select);
        }
        else if (time(NULL) > (force_select_t + select_timeout_sec)) {
            // force timeout
            select_tv.tv_sec = 0; // seconds
            select_tv.tv_usec = 1;
        }
        else {
            select_tv.tv_sec = select_timeout_sec; // seconds
            select_tv.tv_usec = 0;
        }

        if (-1 == (select_rc = select(fdmax+1, &read_fds, &write_fds, NULL, &select_tv))) {
            perror("select");
            // if (EINTR != errno) exit(1);
            // Don't do status here as the death of a child this error
        }
        else if (select_rc == 0) {
            time(&force_select_t);
            // increment timeout up to an hour
            if (select_timeout_sec > 3600) select_timeout_sec = 3600;
            else select_timeout_sec += (select_timeout_sec >= 300 ? 300 : 15);
            printf("\n\n%s\n\n", "--");

            // if current isp has disapeared because of a reread of the conf file,
            // the reconnect should fail
            if (0 == Child_status_flags) {
                // it is safe to reload the isps from the conf file if neccessary
                if (0 != rereadisps(0)) {
                    // read the current isp and set it.  The set checks for validity.
                    // This should stop autoreconnect
                    MICEISP *current_isp_p = get_current_isp();
                    set_current_isp( current_isp_p->shortname);
                    // send isps to clients - neccessary to update the java client's dropdown list
                    check_and_do_command("\002SENDISPS\003", -1, listener,
                                         fdmax, &master, &writemaster);
                }
                // try reconnect if required
                if (0 != is_reconnect_required(listener, fdmax, &master)) {
                    snprintf(data_to_send, CLIENTBUFFSIZE, "\r\n%s\r\n",
                             "0729 ************ RECONNECTING *********");
                    printf("%s", data_to_send);
                    // send to reconnected clients
                    s_queue_cstring(data_to_send, 3, -1, listener, fdmax, &master,  &writemaster);
                    auto_reconnect(listener, fdmax, &master, &writemaster);
                }
                // also reread the ayt from file if neccessary
                rereadayt(0);
            }

            if (0 < (Connect_requests + Reconnect_requests)) {
                // run the oneshot info script if there is a connection
                s_queue_do_info_oneshot(0, -1, listener, fdmax, &master, &writemaster);
                s_queue_ayt(0, -1, listener, fdmax, &master, &writemaster);
            }

            /* status */
            s_queue_status(0, 0, listener, fdmax, &master, &writemaster);

            // temporary
            // s_queue_ayt(0, -1, listener, fdmax, &master, &writemaster); // for client buffer check
            // temporary



            // for inetd server type, exit if no clients and no connector program
            // after 2 * SELECTTIMEOUTSEC
            if ((SERVER_INETD == Params2p->servertype)
            // if ((SERVER_INETD == server_type)
                && (select_timeout_sec >= (Params2p->selecttimeout.tv_sec * 2))
                && (0 >= get_client_count())
                && (Child_pid == getpid())
                && (Info_oneshot_pid == getpid())) {
                exit(0);
            }
        }
        else {
            /* do the recv's first so that new background clients have chance to
             to send DONOTSEND */

            for(fd_i = 0; fd_i <= fdmax; fd_i++) {
                // run through the existing connections looking for data to read
                if (FD_ISSET(fd_i, &read_fds)) { /* data coming in */
                    if (fd_i == listener) {
                        // handle new connections
                        addrlen = sizeof(remoteaddr);
                        if ((newfd = accept(listener, &remoteaddr, &addrlen)) == -1) {
                            perror("accept");
                        }
                        else {
                            // first try to create client data malloc
                            // if error, close the socket
                            if (NULL == (client_data_ptr = create_client_data(newfd))) {
                                printf("Couldn't create data for new client [%d]\n", newfd);
                                // remove the fd
                                close(newfd);
                            }
                            else {
                                printf("client ptr for new client = %p\n", client_data_ptr);

                                FD_SET(newfd, &master); // add to master set
                                if (newfd > fdmax) {    // keep track of the maximum
                                    fdmax = newfd;
                                }

                                {
                                    int flag;
                                    // flag = fcntl(newfd, F_GETFD);
                                    flag = fcntl(newfd, F_SETFD, 1L);
                                    flag = fcntl(newfd, F_GETFD);

                                    // set non-blocking
                                    if (-1 == (fcntl(newfd, F_SETFL, O_NONBLOCK))) { // non-blocking
                                        perror("newfd non-block");
                                    }
                                    // flag = fcntl(newfd, F_GETFL);
                                }
                                {
                                    struct linger sock_linger;
                                    int linger_size = sizeof(struct linger);
                                    if (-1 == getsockopt(newfd, SOL_SOCKET, SO_LINGER,
                                                         &sock_linger, &linger_size)) {
                                        perror("getsockopt");
                                        // exit(1);
                                    }
                                    else {
                                        printf("linger %d:%d\n", sock_linger.l_onoff, sock_linger.l_linger);

                                        if((0 == sock_linger.l_onoff) || (500 > sock_linger.l_linger)) {
                                            sock_linger.l_onoff = 1;
                                            sock_linger.l_linger = 500;
                                            if (-1 == setsockopt(newfd, SOL_SOCKET, SO_LINGER,
                                                                 &sock_linger, linger_size)) {
                                                perror("setsockopt");
                                                // exit(1);
                                            }
                                            else {
                                                printf("linger %d:%d\n", sock_linger.l_onoff, sock_linger.l_linger);
                                            }
                                        }
                                    }
                                }

                                printf("\n\nserver:  new connection from %s on "
                                       "socket %d\n", inet_ntoa(remoteaddr.sin_addr), newfd);

                                //  get the client data pointer
                                // create client data
                                // client_data_ptr = create_client_data(newfd);
                                // printf("client ptr for new client = %p\n", client_data_ptr);

                                snprintf(data_to_send, CLIENTBUFFSIZE, "\r\n\r\n%s %s\r\n",
                                         "0063", Mice_server_version);
                                /* send to new client only */
                                s_queue_cstring(data_to_send, 1, newfd, listener, fdmax, &master,
                                                &writemaster);

                                // java client not ready at this point
                                // Use a command to do this
                                // Send server name for Java client
                                snprintf(data_to_send, CLIENTBUFFSIZE, "%s Server:  %s\r\n", SERVERNAMEMSG, myname_p);
                                // send to new client only
                                s_queue_cstring(data_to_send, 1, newfd, listener, fdmax, &master,
                                                &writemaster);

                                snprintf(data_to_send, CLIENTBUFFSIZE, "\r\n\r\n%s %s (%s)\r\n",
                                         "0062 Hi there", client_data_ptr->nameptr,
                                         client_data_ptr->ip_addressptr);
                                /* send to new client only */
                                s_queue_cstring(data_to_send, 1, newfd, listener, fdmax, &master,
                                                &writemaster);

                                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s\r\n", "0103 This is", myname_p);
                                // send to new client only
                                s_queue_cstring(data_to_send, 1, newfd, listener, fdmax, &master,
                                                &writemaster);


                                /* Send the string to identify the are you alive method used by the
                                 client.  This includes telnet IAC DO TM */
                                s_queue_ayt(1, newfd, listener, fdmax, &master, &writemaster);


                                /* There is a problem here.  The oneshot info script launches a client
                                 which is detected as new here. Then the status function launches the
                                 oneshot script which launches a client ....
                                 */


                                /* send information about commands */
                                snprintf(data_to_send, CLIENTBUFFSIZE,
                                         "\r\n0061 Commands:  %s  %s  %s  %s  %s  %s  %s\r\n\r\n",
                                         "connect", "reconnect", "disconnect", "isps", "status", "BANG", "quit");
                                s_queue_cstring(data_to_send, 1, newfd, listener, fdmax,
                                                &master, &writemaster);
                            }
                        }
                    }
                    else {
                        // get the client data ptr
                        if (NULL != (client_data_ptr = create_client_data(fd_i))) {
                            // can NULL happen here?
                            // printf("client ptr for current client [%d] = %p offset = %d size = %d\n",
                            //       fd_i,
                            //       client_data_ptr,
                            //       client_data_ptr->buffptr - client_data_ptr->buff,
                            //       client_data_ptr->buffsize);
                            // handle data from a client
                            if (0 >= (nbytes = recv(fd_i,client_data_ptr->buffptr,
                                                    client_data_ptr->buffsize, MSG_NOSIGNAL))) {
                                // got error or connection closed by client
                                if (0 == nbytes) {
                                    // connection closed
                                    printf("server:  socket [%d] hung up\n", fd_i);
                                } else {
                                    perror("recv");
                                }
                                close(fd_i); // bye!
                                FD_CLR(fd_i, &master); // remove from master set
                                FD_CLR(fd_i, &writemaster); // remove from master set
                                /* update maximum */
                                if (fd_i >= fdmax) --fdmax;
                                /* decrement connected and remove the client index */
                                client_data_ptr = remove_client_data(fd_i);
                                // printf("client ptr for dead client = %p\n", client_data_ptr);
                                conditional_disconnect();
                            }
                            else {
                                // we got some data from a client
                                check_for_client_newline(nbytes, client_data_ptr,
                                                         fd_i, listener, fdmax,
                                                         &master, &writemaster);
                            }
                        }
                    }
                    // reset timer to minimum when data received
                    // this allows an enter in the client to force a timeout after a long sleep
                    select_timeout_sec = Params2p->selecttimeout.tv_sec;
                }
            } /* end of recv for */

            for(fd_i = 0; fd_i <= fdmax; fd_i++) {

                /* do sends here */
                if (FD_ISSET(fd_i, &write_fds)) { /* there is data to send */
                    s_queue_send(fd_i, listener, &master, &writemaster);
                }
            }
        } /* end of select */
        if (2 == (Child_status_flags & 3)) {
            Child_status_flags &= ~2;
            set_online_flag(0);

            conditional_disconnect();

            // kill background clients
            kill_donotsend_clients(listener, fdmax, &master, &writemaster);

            snprintf(data_to_send, CLIENTBUFFSIZE, "6917 ****** %s ******\r\n", "Connector program has died");
            s_queue_cstring(data_to_send, 0, -1, listener, fdmax, &master,  &writemaster);
            // Can't send to connected and reconnect clients as connected flags have been reset
            if (0 != is_reconnect_required(listener, fdmax, &master)) {
                snprintf(data_to_send, CLIENTBUFFSIZE, "6018 ****** %s ******\r\n", "Wait for reconnection");
                printf("%s\n", data_to_send);
                // send to reconnected clients
                s_queue_cstring(data_to_send, 3, -1, listener, fdmax, &master,  &writemaster);
            }
            force_select_t = 0;
        }

    } /* for loop */

    return 0;
}


void s_queue_ayt(int client_flag, int client_fd, int listener,
                 int fdmax, fd_set *masterptr, fd_set *writemasterptr)
{
    /* send a string to:
     client_flag == 0: all clients
     client_flag == 1: only  to client_fd

     Send the string to identify the are you alive method used by the
     client.  This includes telnet IAC DO TM
     */
    int fd_j;

    if (1 == client_flag) {
        s_queue_ayt_2(client_fd, listener, fdmax, masterptr, writemasterptr);
    }
    else if (0 == client_flag) {
        for(fd_j = 0; fd_j <= fdmax; fd_j++) {
            s_queue_ayt_2(fd_j, listener, fdmax, masterptr, writemasterptr);
        }
    }
}



void s_queue_ayt_2(int client_fd, int listener, int fdmax,
                   fd_set *masterptr, fd_set *writemasterptr)
{
    int slen; // for s_queue_nbytes
    int type; // client type
    int count;
    int aytstatus;
    char data_to_send[CLIENTBUFFSIZE]; /* buffer to send data to clients */
    if (FD_ISSET(client_fd, masterptr)) {
        if (client_fd != listener) {
            aytstatus = get_ayt_status(client_fd, &type, &count);
            if (-1 == aytstatus) {
                // non-responding client
                // get_ayt_status checks for valid client data pointer
                syslog(LOG_USER | LOG_WARNING, "Client %s (%s) [%d] seems to have disappeared.",
                       get_client_name(client_fd), get_client_address(client_fd), client_fd);
                check_and_do_command("quit", client_fd, listener,
                                     fdmax, masterptr, writemasterptr);
            }
            else if (1 == aytstatus) {
                switch(type) {
                case 1: // console
                    snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s [%d] count = %d\r\n", AYTCONSOLE,
                             "Server checking for console client response", client_fd, count);
                             // "Querying \"Are you there console?\"", client_fd, count);
                    break;
                case 2: // java
                    snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s [%d] count = %d\r\n", AYTJAVA,
                             "Server checking for Java client response", client_fd, count);
                    break;
                case 3: // telnet
                    snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s [%d] count = %d\r\n", AYTGENERAL,
                             "Server checking for telnet client response  \377\375\006 ", client_fd, count);
                    break;
                default: // unknown
                    snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s [%d] count = %d\r\n", AYTGENERAL,
                             "Server checking for client response  \377\375\006 ", client_fd, count);
                    break;
                }

                slen = strlen(data_to_send);
                s_queue_nbytes(data_to_send, slen, 1, client_fd, listener, fdmax,
                               masterptr, writemasterptr, 1);
                printf("%s", data_to_send);

                // Have to be really careful here to make sure that background clients
                // have received the AYT.
                increment_ayt_count(client_fd);
            }
        }
    }
}




void s_queue_cstring(char *s, int client_flag, int client_fd, int listener,
                     int fdmax, fd_set *masterptr, fd_set *writemasterptr)
{
    /* send a string to:
     client_flag == 0: all clients
     client_flag == -1: not to client_fd
     client_flag == 1: only  to client_fd
     client_flag == 2: all connected clients
     client_flag == 3: all reconnected clients
     client_flag == anything else: all clients
     Never send to listener.
     */
    int slen;
    slen = strlen(s);
    s_queue_nbytes(s, slen, client_flag, client_fd, listener, fdmax,
                   masterptr, writemasterptr, 0);
}




void s_queue_nbytes(char *s, int nbytes, int client_flag, int client_fd, int listener,
                    int fdmax, fd_set *masterptr, fd_set *writemasterptr, int sendoverride)
{
    /* send a nbytes to:
     client_flag == 0: all clients
     client_flag == -1: not to client_fd
     client_flag == 1: only  to client_fd
     client_flag == 2: all connected clients
     client_flag == 3: all reconnected clients
     client_flag == anything else: all clients
     Never send to listener.
     sendoverfide == 1 to override donotsend.  Used for AYT
     */

    CLIENT_DATA * client_data_ptr;
    char *client_buffp;
    int fd_j;
    int not_client = - 1; /* -1 will not be found in for loop */
    int (*how_connected)();
    how_connected = is_client_reconnected; // for case 3, reconnected clients
    switch (client_flag) {
    case 1: /* send only to client_fd */
        if (FD_ISSET(client_fd, masterptr)) {
            if (client_fd != listener) {
                if (NULL != (client_data_ptr = get_client_data(client_fd))) {
                    if ((0 == client_data_ptr->donotsend) || (0 != sendoverride)) {
                        /* This client accepts data */
                        /* get the client's data send buffer */
                        if (NULL != (client_buffp = s_queue_alloc(client_data_ptr, nbytes))) {
                            /* client_buffp points to the client's send buffer */
                            memcpy(client_buffp, s, nbytes);
                            client_data_ptr->bytes_notsent += nbytes;
                            FD_SET(client_fd, writemasterptr); /* add to send master set */
                        }
                    }
                    // we don't want this
                    // else  FD_CLR(client_fd, writemasterptr); /* remove from send master set */
                }
            }
        }
        break;

    case 2:
    how_connected = is_client_connected;  // connected clients
    case 3:
        for(fd_j = 0; fd_j <= fdmax; fd_j++) {
            // send to everyone connected or reconnected
            if (FD_ISSET(fd_j, masterptr)) {
                /* except the listener and not client */
                if (fd_j != listener) {
                    if (NULL != (client_data_ptr = get_client_data(fd_j))) {
                        if ((0 == client_data_ptr->donotsend) || (0 != sendoverride)) {
                            if (0 != (*how_connected)(fd_j)) {
                                /* get the client's data send buffer */
                                if (NULL != (client_buffp = s_queue_alloc(client_data_ptr, nbytes))) {
                                    /* client_buffp points to the client's send buffer */
                                    memcpy(client_buffp, s, nbytes);
                                    client_data_ptr->bytes_notsent += nbytes;
                                    FD_SET(fd_j, writemasterptr); /* add to send master set */
                                }
                            }
                        }
                        // else  FD_CLR(fd_j, writemasterptr); /* remove from send master set */
                    }
                }
            }
        }
        break;

    case -1: /* not to client_fd */
        not_client = client_fd;
    default:
        for(fd_j = 0; fd_j <= fdmax; fd_j++) {
            // send to everyone!
            if (FD_ISSET(fd_j, masterptr)) {
                /* except the listener and not client */
                if (fd_j != listener && fd_j != not_client) {
                    if (NULL != (client_data_ptr = get_client_data(fd_j))) {
                        if ((0 == client_data_ptr->donotsend) || (0 != sendoverride)) {
                            /* This client accepts data */
                            /* get the client's data send buffer */
                            if (NULL != (client_buffp = s_queue_alloc(client_data_ptr, nbytes))) {
                                /* client_buffp points to the client's send buffer */
                                memcpy(client_buffp, s, nbytes);
                                client_data_ptr->bytes_notsent += nbytes;
                                FD_SET(fd_j, writemasterptr); /* add to send master set */
                            }
                        }
                        // we don't want this
                        // else  FD_CLR(fd_j, writemasterptr); /* remove from send master set */
                    }
                }
            }
        }
    }
}


char *s_queue_alloc(CLIENT_DATA * client_data_ptr, int nbytes)
{
    /* return a pointer to the place in the buffer where new data can be added,
     after checking/allocating the send buffer size */
    /* note client_data_ptr->sendbuff could be NULL but OK for realloc. Test
     on the buffer size is sufficient */
    char * newbuffp = NULL;
    if (NULL != client_data_ptr) {
        if (nbytes <= (client_data_ptr->sendbuffsize
                       - client_data_ptr->bytes_sent
                       - client_data_ptr->bytes_notsent)) {
            newbuffp = client_data_ptr->sendbuff
                + client_data_ptr->bytes_sent
                + client_data_ptr->bytes_notsent;
        }
        else if (NULL != (newbuffp = realloc(client_data_ptr->sendbuff,
                                             (client_data_ptr->bytes_sent
                                              + client_data_ptr->bytes_notsent
                                              + nbytes) * sizeof(char)))) {
            client_data_ptr->sendbuff = newbuffp;
            newbuffp = client_data_ptr->sendbuff + client_data_ptr->bytes_sent
                + client_data_ptr->bytes_notsent;
            client_data_ptr->sendbuffsize = (client_data_ptr->bytes_sent
                                             + client_data_ptr->bytes_notsent
                                             + nbytes);
            /* temporary */
            printf("send queue realloc for [%d] %d\n", client_data_ptr->fd,
                   client_data_ptr->sendbuffsize);
            /* sleep(3); */
        }
        /* client_data_ptr->sendbuff could still be NULL or too small */
    }
    return newbuffp; /* could be NULL */
}


void s_queue_send(int client_fd, int listener, fd_set *masterptr, fd_set *writemasterptr)
{
    /* send nbytes */
    CLIENT_DATA * client_data_ptr;
    int bytes_sent;
    // printf("%s\n", "In s_queue_send");
    if (FD_ISSET(client_fd, masterptr)) {
        if (client_fd != listener) {
            if (NULL != (client_data_ptr = get_client_data(client_fd))) {
                if (-1 == (bytes_sent = send(client_fd, client_data_ptr->sendbuff
                                             + client_data_ptr->bytes_sent,
                                             client_data_ptr->bytes_notsent, MSG_NOSIGNAL))) {
                    perror("s_queue_send");
                    /* forget everything */
                    client_data_ptr->bytes_sent = client_data_ptr->bytes_notsent = 0;
                    FD_CLR(client_fd, writemasterptr); /* remove from send master set */
                }
                else {
                    client_data_ptr->bytes_sent += bytes_sent;
                    client_data_ptr->bytes_notsent -= bytes_sent;
                    /* if bytes_notsent < 0, something is wrong */
                    if (0 >= client_data_ptr->bytes_notsent) {
                        /* all gone */
                        client_data_ptr->bytes_sent = client_data_ptr->bytes_notsent = 0;
                        FD_CLR(client_fd, writemasterptr); /* remove from send master set */
                    }
                }
            }
        }
    }
    else {
        /* not in the master set, so remove from write master set */
        FD_CLR(client_fd, writemasterptr); /* remove from send master set */
    }
    // printf("%s\n", "End of s_queue_send");
}


void s_queue_status(int client_flag, int client_fd, int listener, int fdmax,
                    fd_set *masterptr, fd_set *writemasterptr)
{
    /* send status information to one or all clients */
    /* send to:
     client_flag == 0: all clients
     client_flag == -1: not to client_fd
     client_flag == 1: only  to client_fd
     Never send to listener
     */
    static char myname[MYNAMESIZE] = ""; // null string neccessary for first time test
    static char dashes_top[2][31] = {"", ""}; // for formatted string
    static char dashes_bottom[61] = ""; // for formatted string
    int about_ctr, to_ctr, client_count;
    CLIENT_DATA * client_data_ptr;
    char data_to_send[CLIENTBUFFSIZE]; /* buffer to send data to clients */

    if ('\0' == *myname) {
        // first time
        // this is formatting for the top and bottom of the status display
        int dash_count = 1;  // number of dashes around "time on server" string
        int dash_extra = 0;
        strcpy(myname, getservername());

        // strcpy(myname, "01234567890123456789012345678901234567890123456789");
        // printf("strlen myname %d\n", strlen(myname));
        *(myname + 30) = '\0';  // not too long for display
        // calculate number of dashes around of "time on server" string
        // careful, size_t is unsigned
        dash_count = 60 - (int)(strlen(getnow_s()) + strlen(myname) + 8);
        // printf("dash count %d\n", dash_count);
        if (2 > dash_count) dash_count = 2;
        // printf("dash count %d\n", dash_count);
        if (0 != dash_count % 2) dash_extra = 1;
        dash_count /= 2;
        // printf("dash count %d\n", dash_count);
        // printf("dash extra %d\n", dash_extra);
        // if (0 >= dash_count) dash_count = 1;
        // sleep(2);
        memset(dashes_top[0], '-', dash_count);
        *(dashes_top[0] + dash_count) = '\0';
        memset(dashes_top[1], '-', dash_count + dash_extra);
        *(dashes_top[1] + dash_count + dash_extra) = '\0';
        memset(dashes_bottom, '-', 60);
        *(dashes_bottom + 60) = '\0';
    }


    // check for demo server
    if (!strcmp(DEMOLOGIDENT, Params2p->logident)) {
        snprintf(data_to_send, CLIENTBUFFSIZE, "\r\n0010 ---------  %s  ---------\r\n",
                 "Demonstration server");
        s_queue_cstring(data_to_send, client_flag, client_fd, listener, fdmax, masterptr, writemasterptr);
    }
    // end of demo server check

    snprintf(data_to_send, CLIENTBUFFSIZE, "\r\n%s %s  %s on %s  %s\r\n",
             STARTOFSTATUS, dashes_top[0], getnow_s(), myname, dashes_top[1]);
    s_queue_cstring(data_to_send, client_flag, client_fd, listener, fdmax, masterptr, writemasterptr);


    if (9 > (client_count = get_client_count())) {
        // don't send this if lots of clients
        /* loop through all the client data structures to collect data */
        for(about_ctr = 0, client_count = 0; about_ctr <= fdmax; about_ctr++) {
            if (FD_ISSET(about_ctr, masterptr)) {
                /* except the listener */
                if (about_ctr != listener) {
                    if (NULL != (client_data_ptr = get_client_data(about_ctr))) {
                    ++client_count;
                    snprintf(data_to_send, CLIENTBUFFSIZE, "%-*s %s (%s) [%d]\r\n",
                             Connect_requests + Reconnect_requests ? 25 : 4,
                             client_data_ptr->connected.flag.connected ? "0008 Connect request:" :
                             (client_data_ptr->connected.flag.reconnected ? "0009 Reconnect request:" : "0007"),
                             client_data_ptr->nameptr, client_data_ptr->ip_addressptr,
                             client_data_ptr->fd);
                    s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                                    fdmax, masterptr, writemasterptr);
                    }
                }
            }
        }
    }

    snprintf(data_to_send, CLIENTBUFFSIZE,
             "%s %-20s %d\r\n%s %-20s %d\r\n%s %-20s %d\r\n",
             CLIENTCOUNT, "Client count:", client_count,
             CONNECTREQUESTS, "Connect requests:", Connect_requests,
             RECONNECTREQUESTS, "Reconnect requests:", Reconnect_requests);
    printf("%s", data_to_send);
    s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                    fdmax, masterptr, writemasterptr);


    /* Child connection information */
    if (Child_pid == getpid()) {
        // MICEISP *current_isp_p = get_current_isp();  // temporary
        snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CONNECTORNOTRUN,
                 "Connector program:", "NOT running");
        s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                        fdmax, masterptr, writemasterptr);
        // current isp - temporary
        /*
         snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CURRENTISPSHORT,
         "Current ISP:", current_isp_p->shortname);
         s_queue_cstring(data_to_send, client_flag, client_fd, listener,
         fdmax, masterptr, writemasterptr);
         */
    }
    else {
        MICEISP *current_isp_p = get_current_isp();
        snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CONNECTORRUN,
                 "Connector program:", "running");
        s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                        fdmax, masterptr, writemasterptr);
        // current isp
        snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CURRENTISPSHORT,
                 "Current ISP:", current_isp_p->shortname);
        s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                        fdmax, masterptr, writemasterptr);
        snprintf(data_to_send, CLIENTBUFFSIZE, "%-*s %s\r\n", 25, CURRENTISPLONG,
                 current_isp_p->longname);
        s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                        fdmax, masterptr, writemasterptr);
        // check for online
        if (1 == get_online_flag()) {
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", LINEON,
                     "Line status:", "online");
            s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                            fdmax, masterptr, writemasterptr);
        }
        else if (-1 == get_online_flag()) {
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", LINEOFF,
                     "Line status:", "offline");
            s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                            fdmax, masterptr, writemasterptr);
        }
    }
    {
        char *msg_ptr;
        msg_ptr = get_message();
        if (0 != strlen(msg_ptr)) {
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", INFOMSG, "Information:", msg_ptr);
            s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                            fdmax, masterptr, writemasterptr);
        }
    }


    /* if sending to only one client */
    if (1 == client_flag) {
        if (NULL != (client_data_ptr = get_client_data(client_fd))) {
            char tempbuff[30]; // For formatting
            sprintf(tempbuff, "%s [%d]:", "This client", client_data_ptr->fd);
            snprintf(data_to_send, CLIENTBUFFSIZE,
                     "%s %-20s %sconnection %srequested\r\n",
                     client_data_ptr->connected.flag.connected ? MYCONNECT :
                     (client_data_ptr->connected.flag.reconnected ? MYRECONNECT : MYNOCONNECT),
                     tempbuff,
                     client_data_ptr->connected.flag.reconnected ? "re" : "",
                     client_data_ptr->connected.both ? "" : "NOT ");
            s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                                    fdmax, masterptr, writemasterptr);
        }
    }
    else if (0 == client_flag) {
        char tempbuff[30]; // For formatting.  Declare before the loop
        for(to_ctr = 0; to_ctr <= fdmax; to_ctr++) {
            if (FD_ISSET(to_ctr, masterptr)) {
                /* except the listener  */
                if (to_ctr != listener) {
                    if (NULL != (client_data_ptr = get_client_data(to_ctr))) {
                        /* a different string for each client */
                        sprintf(tempbuff, "%s [%d]:", "This client", client_data_ptr->fd);
                        snprintf(data_to_send, CLIENTBUFFSIZE,
                                 "%s %-20s %sconnection %srequested\r\n",
                                 client_data_ptr->connected.flag.connected ? MYCONNECT :
                                 (client_data_ptr->connected.flag.reconnected ? MYRECONNECT : MYNOCONNECT),
                                 tempbuff,
                                 client_data_ptr->connected.flag.reconnected ? "re" : "",
                                 client_data_ptr->connected.both ? "" : "NOT ");
                        s_queue_cstring(data_to_send, 1, to_ctr, listener,
                                                fdmax, masterptr, writemasterptr);
                    }
                }
            }
        } /* for */
    }

    snprintf(data_to_send, CLIENTBUFFSIZE, "\r\n0061 %-20s %-7s  %-9s  %-10s\r\n0061 %-20s %-7s  %-9s  %-10s  %-4s\r\n",
             "Commands:", "connect", "reconnect", "disconnect", "", "isps", "status", "BANG", "quit");
    s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                            fdmax, masterptr, writemasterptr);
    snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s\r\n\r\n", ENDOFSTATUS, dashes_bottom);
    s_queue_cstring(data_to_send, client_flag, client_fd, listener, fdmax, masterptr, writemasterptr);

}  /* s_queue_send() */



pid_t s_queue_do_info_oneshot(int client_flag, int client_fd, int listener, int fdmax,
                              fd_set *masterptr, fd_set *writemasterptr)
{
    /* in parent, returns -1 on failure, or child pid on success or already a child */
    /* Data cannot be queued to send in the child as it disappears before the
     the data can be sent in the select loop of the child */
    char data_to_send[CLIENTBUFFSIZE]; /* buffer to send data to clients */
    pid_t fork_rc = -1;
    char *magic_id;
    magic_id = get_magic_id();
    /* child pid is set to parent in zombie handler */
    if (Info_oneshot_pid == getpid()) {
        MICEISP *current_isp_p = get_current_isp();
        if (0 != check_exec_access()) {
            fork_rc = -1;
        }
        else {
            /* run the child */
            fork_rc = fork();
            /* fork_rc = -1; */  /* test */
            if (0 == fork_rc) {
                // this is the child

                printf("In the info child%s\n", Params2p->execinfo);


                // setpgrp();
                setpgid(0,0);

                execl(Params2p->execinfo, Params2p->infoprogram,
                      current_isp_p->shortname, current_isp_p->longname,
                      Params2p->scriptdir, NULL);
                syslog(LOG_USER | LOG_WARNING, "%s did not run", Params2p->execinfo);
                snprintf(data_to_send, CLIENTBUFFSIZE, "\r\n\r\n%s\r\n\r\n",
                         "7906 ****** Info child exec not done ******");
                /* sleep(5); */
                printf("%s", data_to_send);
                sleep(2);  // delay the exit
                _exit(1); /* exit child */
                // potential problem here is this exit occurs before parent sets the
                // Info pid to fork_rc.  Error handler won't reset it.
            }
            else if (-1 == fork_rc) {
                /* child fork failed */
                syslog(LOG_USER | LOG_WARNING, "fork for %s failed", Params2p->execinfo);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s\r\n",
                         "7907 Info program did not start");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                                fdmax, masterptr, writemasterptr);
            }
            else {
                /* parent */
                Info_oneshot_pid = fork_rc;
                snprintf(data_to_send, CLIENTBUFFSIZE, "7001 in parent, Info onshot child pid = %d\r\n",
                         Info_oneshot_pid);
                printf("%s", data_to_send);
            }
        } // access check
    }  // pid check
    else {
        fork_rc = Info_oneshot_pid; // already a child
        printf("%s\n", "info child is already running");
    }
    return fork_rc;
}


pid_t s_queue_do_connect(int client_flag, int client_fd, int listener, int fdmax,
                         fd_set *masterptr, fd_set *writemasterptr) {
    /* in parent, returns -1 on failure, or child pid on success or already a child */
    char data_to_send[CLIENTBUFFSIZE]; /* buffer to send data to clients */
    pid_t fork_rc = -1;
    char *magic_id;
    magic_id = get_magic_id();
    /* child pid is set to parent in zombie handler */
    if (Child_pid == getpid()) {
        // the getpid check is required because of autoreconnect by multiple clients
        // as well as for safety
        MICEISP *current_isp_p = get_current_isp();

        // If we reach this point more than 5 times in 5 minutes, assume runaway.
        // This can occur, for example, if the connect script doesn't run pppd
        // because the isp isn't recognised.

        if (0 != check_for_runaway()) fork_rc = -1;

        // set the global current isp here.
        // auto reconnect should pass the existing isp
        // connect and reconnect should pass the isp from the command line
        // don't fork is isp is invalid - checked against list of valid isps

        else if ('\0' == *(current_isp_p->shortname)) {
            printf("%s\n","******* INVALID ISP ********");
            // Problem - client thinks it is connected
            // set_client_unconnected_both(client_fd);  // that should fix it
            fork_rc = -1;
        }
        else if (0 != check_exec_access()) {
            // if connector program has become non-executable, autoreconnect would loop
            // set_client_unconnected_both(client_fd);
            fork_rc = -1;
        }

        else {
            // run the child
            fork_rc = fork();
            // fork_rc = -1;   // test
            if (0 == fork_rc) {
                // this is the child

                printf("In the connector child %s\n", Params2p->execconnect);

                setpgid(0,0);

                if (-1 == execl(Params2p->execconnect, Params2p->connectprogram,
                                current_isp_p->shortname, current_isp_p->longname,
                                Params2p->scriptdir, NULL)) {
                    perror("child exec");
                }
                syslog(LOG_USER | LOG_WARNING, "%s did not run", Params2p->execconnect);
                snprintf(data_to_send, CLIENTBUFFSIZE, "\r\n\r\n%s\r\n\r\n",
                         "6906 ****** Connector child exec not done ******");
                /* sleep(5); */
                printf("%s", data_to_send);
                sleep(2);  // delay the exit
                _exit(1); /* exit child */
                // Potential problem here.  If this exit occurs before parent has set the Child_pid
                // error handler will not set the Child_pid back.
            }
            else if (-1 == fork_rc) {
                // child fork failed
                // Problem - client thinks it is connected
                set_client_unconnected_both(client_fd); // that should fix it
                set_current_isp("");
                syslog(LOG_USER | LOG_WARNING, "fork for %s failed", Params2p->execconnect);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CONNECTORNOSTART,
                         "Connector program:", "NOT started");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, client_flag, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
            }
            else {
                /* parent */
                Child_pid = fork_rc;
                printf("%s in parent, child pid = %d\n", CONNECTORSTART, Child_pid);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CONNECTORSTART,
                         "Connector program:", "started");
                printf("%s", data_to_send);
                // tell everyone
                s_queue_cstring(data_to_send, 0, -1, listener, fdmax,
                                masterptr, writemasterptr);
                Child_status_flags |= 3; /* bits */
            }
        }  // isp test
    } // pid test

    else {
        fork_rc = Child_pid; /* already a child */
    }

    if (-1 == fork_rc) {
        set_current_isp("");
        if (0 != is_client_connected_either(client_fd)) {
            check_and_do_command("disconnect", client_fd, listener,
                                 fdmax, masterptr, writemasterptr);
        }
    }
    return fork_rc;
}


int check_for_runaway(void)
{
    // If called more than 5 time in 5 minutes, return non-zero
    // else return 0
    static time_t runs[5] = {0,0,0,0,0};
    static int run_ctr = 0;
    time_t now;
    int rc = 1;  // 0 is OK, set to 1 if runaway
    int ctr;
    // now
    now = time(NULL);
    // check if last 5 runs have been within 5 minutes
    for (ctr = 0; ctr < 5; ++ctr) {
        printf("Runaway check %ld  %ld\n",
               (long)*(runs + ctr), (long)now - (long)*(runs + ctr));  // debug
    }
    for (ctr = 0; ctr < 5; ++ctr) {
        if ((*(runs + ctr) + 300) < now) {
            rc = 0;
            break;
        }
    }
    if (0 == rc) {
        // Increment counter if connect allowed.
        // This allows the user to keep on trying to connect
        // and the failures don't update the stored times
        run_ctr = (run_ctr + 1) % 5;
        // store latest time
        *(runs + run_ctr) = now;
    }
    return rc;
}




void sigchld_handler(int s) {
    /* zombie handler */
    pid_t pid;
    // pid = waitpid(-1, &Child_pid_status, WNOHANG);
    while (0 < (pid = waitpid(-1, NULL, WNOHANG))) {
        // loop in case both children enter only one instance of the handler
        if (pid == Child_pid) {
            // printf("zombie handler: connector child %d died\n", pid);
            Child_pid = getpid();  /* for testing - a child can't be this */
            set_all_clients_unconnected();
            set_message(NULL); // clear the message buffer
            /* flag as not connected */
            Child_status_flags &= ~1;
            /* if the child has died, we need to kill the info child */
            /* ensure that the child pid has been reset to getpid() first */
            set_online_flag(0);

            // don't do here, do at bottom of select loop depending on
            // Child_status_flags
            // conditional_disconnect();
        }
        else if (pid == Info_oneshot_pid) {
            // printf("zombie handler: info oneshot child %d died\n", pid);
            Info_oneshot_pid = getpid();  /* for testing - a child can't be this */
        }
    }
}


void conditional_disconnect(void)
{
    // This is called from the signal handler for child death. Not any longer.
    // check the connected counter
    if (0 >= (Connect_requests + Reconnect_requests)) {

        // unset the the global current isp here
        // if there is a reconnect, isp will not be unset
        // note isp may be NULL so check
        set_current_isp("");

        if (Child_pid != getpid()) {
            do_disconnect(Child_pid, Params2p->execkillconnect, Params2p->killconnectprogram);
        }
        if (Info_oneshot_pid != getpid()) {
            do_disconnect(Info_oneshot_pid, Params2p->execkillinfo, Params2p->killinfoprogram);
        }
    }
}


// void do_disconnect(pid_t pid_to_kill)
void do_disconnect(pid_t pid_to_kill, char *exec_p, char const  *program_p)
{
    // Need to remove the printfs as this can be called from signal handler - not any longer
    /* in parent, returns -1 on failure, or child pid on success or already a child */
    pid_t fork_rc;
    char pid_to_kill_string[NUMBERBUFFSIZE]; /* make it big enough */


    /* change the pid_to_kill to a string */
    snprintf(pid_to_kill_string, NUMBERBUFFSIZE, "%ld", (long) pid_to_kill);  /* What is a pid_t */
    printf("killing %ld  %s\n", (long) pid_to_kill, pid_to_kill_string);
    /* sleep(5); */
    /* run the child */
    fork_rc = fork();
    if (0 == fork_rc) {
        /* this is the child */
        printf("%s\n", "In the kill child");

        puts(exec_p);

        setpgid(0,0);

        if (-1 == execl(exec_p, program_p, pid_to_kill_string, NULL)) {
            perror("kill child exec");
        }
        syslog(LOG_USER | LOG_WARNING, "%s did not run", exec_p);
        printf("%s\n", "******* kill child exec not done **************");
        _exit(1); /* exit child */
    }
    else if (-1 == fork_rc) {
        /* child fork failed */
        syslog(LOG_USER | LOG_WARNING, "fork for %s failed", exec_p);
        printf("%s\n", "********* Kill child fork failed ***********");
    }
    else {
        /* parent */
        printf("in parent, kill child pid = %d\n", fork_rc);
    }
}




void check_for_client_newline(int nbytes, CLIENT_DATA *client_data_ptr, int client_fd,
                          int listener, int fdmax, fd_set *masterptr, fd_set *writemasterptr)
{
    static char cbuff[CLIENTBUFFSIZE + 1];
    // char data_to_send[CLIENTBUFFSIZE]; /* buffer to send data to clients */
    char *cptr = NULL;
    char *nl_ptr;
    size_t len_data, len_line, len_data_notchecked;
    int iscommand; /* will be zero if the line is not a command */
    int ctr;

    len_data = (client_data_ptr->buffptr - client_data_ptr->lineptr) + nbytes;  /* data from start of current line */
    len_data_notchecked = nbytes; /* new data */

    /* scan new data for telnet commands.  Synch (DT) can be on its own without being proceeded
     by the escape character */
    for (ctr = 240; ctr <= 255; ++ctr) {
        if (NULL != memchr(client_data_ptr->buffptr, ctr,  nbytes)) {
            printf("%s\n"," ************* telnet command ************************");
            if (1 == check_telnet(client_data_ptr->lineptr, len_data)) {
                // snprintf(data_to_send, CLIENTBUFFSIZE, "8500 Response to [%d] telnet AYT\r\n", client_fd);
                // printf("%s", data_to_send);
                // s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                //                masterptr, writemasterptr);
                check_and_do_command("\002TELNETCLIENTAYT\003", client_fd,
                                     listener, fdmax, masterptr, writemasterptr);
            }
        }
    }

    while (0 < len_data_notchecked) {
        if (NULL != (nl_ptr = memchr(client_data_ptr->buffptr, '\n',  len_data_notchecked))) {
            /* new line found */
            len_line = nl_ptr - client_data_ptr->lineptr + 1;
            // printf("len_line %d\n", len_line);
            // printf("nlp %p cdp %p\n", nl_ptr, client_data_ptr->lineptr);
            memcpy(cbuff, client_data_ptr->lineptr, len_line);
            *(cbuff + (len_line - 1)) = '\0';
        
            /* there may be a \r */
            if (NULL != (cptr = strchr(cbuff, '\r'))) {
                /* remove it */
                *cptr = '\0';
            }
            cptr = cbuff;  /* restore to start of string */
            printf("success: %s\n", cptr);


            /* check for command here */
            /* CAREFUL.  There could be two commands in one recv and the first could be
             a QUIT that would destroy the clients data */
            if (!(FD_ISSET(client_fd, masterptr))) return;
            else {
                // iscommand = check_and_do_command(cbuff, client_data_ptr, client_fd, listener,
                //                      fdmax, masterptr, writemasterptr);
                iscommand = check_and_do_command(cbuff, client_fd, listener,
                                     fdmax, masterptr, writemasterptr);
                if (!(FD_ISSET(client_fd, masterptr))) return;
            }


            /* if it wasn't a command, send to everyone except self */
            /* don't send line conisting enirely of spaces -  this is because of the telnet stuff */
            /* but send empty lines */
            /********************
            // Don't send non-commands on to clients anymore
            if ((0 == iscommand) && ((strlen(cbuff) != strspn(cbuff, " ")) || (0 == strlen(cbuff)))) {
                snprintf(data_to_send, CLIENTBUFFSIZE, "[%d] %s\r\n", client_fd, cbuff);
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, -1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
            }
            *********************/


            /* Is there more in the buffer */

            /* move ptr to after the \n */
            len_data_notchecked = len_data - len_line;
            /* if there is no more in the buffer after a \n, the set ptr to
             beginning of the buffer */
            if (0 >= len_data_notchecked) {
                client_data_ptr->buffsize = sizeof(client_data_ptr->buff);
                client_data_ptr->lineptr = client_data_ptr->buffptr = client_data_ptr->buff;
            }
            else {
                /* there will be another loop */
                client_data_ptr->buffptr = client_data_ptr->lineptr = nl_ptr + 1;  /* start of next line */
                len_data = len_data_notchecked;  /* data from start of current line */
            }
        }
        else {
            client_data_ptr->buffptr += nbytes;
            client_data_ptr->buffsize -= nbytes;
            len_data_notchecked = 0;
            /* leave the lineptr alone */
        }
    } /* end while */

    /* buffptr now points to char after end of new data.
     This could be past the end of the buffer */
    if ((client_data_ptr->buffptr - client_data_ptr->buff) >= CLIENTBUFFSIZE) {
        /* buffer full, abandon */
        client_data_ptr->buffsize = sizeof(client_data_ptr->buff);
        client_data_ptr->lineptr = client_data_ptr->buffptr = client_data_ptr->buff;
        printf("%s\n", "receive buffer overflow - don't panic");
    }
}

int check_telnet(char *linep, int len_data)
{
    // set telnet commands to spaces
    char *escapep;
    int len_escape;
    int found;  // for outer loop breaking
    int ayt = 0; // set to 1 when timing mark found
    char tests[4] = "\xFF";
    int testi;
    while (NULL != (escapep = memchr(linep, '\xFF',  len_data))) {
        found = 0;
        len_escape = len_data - (escapep - linep);
        /* look for 3 char command */
        if (len_escape >= 3) {
            for (testi = 251; testi <= 254; ++testi) {
                *(tests + 1) = (unsigned char)testi;
                // if (!strncmp(escapep, tests, 2)) {
                if (!memcmp(escapep, tests, 2)) {
                    printf("%s %d %d\n", "******** telnet 3 char command found",
                           (unsigned char)*(escapep + 1),
                           (unsigned char)*(escapep + 2));
                    // *escapep = *(escapep + 1) = *(escapep + 2) = ' ';
                    // *escapep = *(escapep + 1) = *(escapep + 2) = 'x';
                    // continue;  // while.  look for more

                    // check here for response to timing mark sent by server.
                    //  \377\375\006
                    if (6 == (unsigned char)*(escapep + 2)) {
                        printf("%s\n", "Telnet timing mark received");
                        ayt = 1;

                    }
                    *escapep = *(escapep + 1) = ' ';
                    *(escapep + 2) = '\n';
                    found = 1;
                    break;  // for
                }
            }
            if (found) continue;  // while.  look for more
        }
        /* look for 2 char command */
        if (len_escape >= 2) {
            *(tests + 1) = 246;
            if (!strncmp(escapep, tests, 2)) {
                printf("%s\n", "******** telnet AYT found");
            }
            // also double IAC
            *(tests + 1) = 255;
            if (!strncmp(escapep, tests, 2)) {
                printf("%s\n", "******** telnet double IAC found");
            }
            // wipe it out, including double IAC
            printf("%s %d\n", "******** telnet 2 char command found",
                   (unsigned char)*(escapep + 1));
            *escapep = ' ';
            *(escapep + 1) = '\n';
            // *escapep = *(escapep + 1) = ' ';
            // *escapep = *(escapep + 1) = 'y';
            continue;  // while.  look for more
        }

        // At this point, no more 2 or 3 char commands, but the could be 1 char commands, eg DT.
        // Or partial 2 (IAC) or 3 char commands if the whole of the command hasn't been sent.
        break;
    } //end while
    // remove any single DT
    /*
    for (testi = 0; testi < len_data; ++testi) {
        if (242 == (unsigned char)*(linep + testi)) {
            printf("%s\n", "******** telnet synch char found");
            *(linep + testi) = ' ';
        }
        }
        */
    return ayt;  // 1 if timing mark found
}



int check_and_do_command(char *client_string, int client_fd,
                          int listener, int fdmax, fd_set *masterptr, fd_set *writemasterptr)
{
    char data_to_send[CLIENTBUFFSIZE]; /* buffer to send data to clients */
    int iscommand = 1; /* reset to 0 if not a keyword */

    // remove leading spaces, ready for telnet command processing
    while (' ' == *client_string && '\0' != *client_string) ++client_string;



    // online, offline, info will be most frequent so check first

    if (!strncasecmp("\002ONLINE\003", client_string, 8)) {
        // online message from info message client
        if (Child_pid == getpid()) set_online_flag(0);  // no connector
        else if (is_server_id_ok(client_fd)) {
            // Online_flag = 1;
            set_online_flag(1);
            if (0 != cmp_online_flag()) {
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", LINEON2,
                         "Status:", "online");
                // tell everyone
                s_queue_cstring(data_to_send, 0, -1, listener, fdmax, masterptr, writemasterptr);
                // s_queue_cstring(data_to_send, 2, -1, listener, fdmax, masterptr, writemasterptr);
                // s_queue_cstring(data_to_send, 3, -1, listener, fdmax, masterptr, writemasterptr);
            }
        }
    }

    else if (!strncasecmp("\002OFFLINE\003", client_string, 9)) {
        // online message from info message client
        if (Child_pid == getpid()) set_online_flag(0);
        else if (is_server_id_ok(client_fd)) {
            // Online_flag = -1;
            set_online_flag(-1);
            if (0 != cmp_online_flag()) {
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", LINEOFF2,
                         "Status:", "offline");
                // tell everyone
                s_queue_cstring(data_to_send, 0, -1, listener, fdmax, masterptr, writemasterptr);
                // s_queue_cstring(data_to_send, 2, -1, listener, fdmax, masterptr, writemasterptr);
                // s_queue_cstring(data_to_send, 3, -1, listener, fdmax, masterptr, writemasterptr);
            }
        }
    }


    else if (!strncasecmp("\002INFO\003", client_string, 6)) {
        // info message from info message client
        char *ptr, *msg_ptr;
        msg_ptr = client_string + 6;
        if (is_server_id_ok(client_fd)) {
            // remove multi-lines if there is an odd \r or \n
            if (NULL != (ptr = strchr(msg_ptr, '\r'))) *ptr = '\0';
            if (NULL != (ptr = strchr(msg_ptr, '\n'))) *ptr = '\0';
            set_message(msg_ptr);
            if (0 != cmp_message()) {
                // send an immediate message to connected and reconnected clients
                // if the information buffer has changed.
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", INFOMSG2,
                         "Information:", msg_ptr);
                // tell everyone
                s_queue_cstring(data_to_send, 0, -1, listener, fdmax, masterptr, writemasterptr);
                // s_queue_cstring(data_to_send, 2, -1, listener, fdmax, masterptr, writemasterptr);
                // s_queue_cstring(data_to_send, 3, -1, listener, fdmax, masterptr, writemasterptr);
            }
        }
    }

    else if (!strcasecmp("\002CONSOLECLIENTAYT\003", client_string)) {
        int old_count;
        old_count = reset_ayt_count(client_fd, CLIENTCONSOLE);
        printf("%s %d\n", "Console client AYT received", old_count);
    }

    else if (!strcasecmp("\002JAVACLIENTAYT\003", client_string)) {
        int old_count;
        old_count = reset_ayt_count(client_fd, CLIENTJAVA);
        printf("%s %d\n", "Java client AYT received", old_count);
    }

    else if (!strcasecmp("\002TELNETCLIENTAYT\003", client_string)) {
        int old_count;
        old_count = reset_ayt_count(client_fd, CLIENTTELNET);
        printf("%s %d\n", "Telnet client AYT received", old_count);
    }



    else if ((!strcasecmp("CONNECT", client_string)) || (!strncasecmp("CONNECT ", client_string, 8))) {
        printf("connection requested %u\n", is_client_connected_either(client_fd));

        syslog(LOG_USER | LOG_INFO, "connection request from %s [%s]",
               get_client_name(client_fd), get_client_address(client_fd));


        if (Child_pid != getpid()) {
            // connector running

            /* check if already requested connection */
            if (0 != is_client_connected(client_fd)) {
                char tempbuff[30]; // For formatting
                sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                         MYCONNECTALREADY, tempbuff, "connection already requested");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
            }
            else {
                char tempbuff[30]; // For formatting
                sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                         MYCONNECTREQ, tempbuff, "connection requested");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
                set_client_connected(client_fd);
                set_client_unreconnected(client_fd);
            }
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CONNECTORALREADY,
                     "Connector program:", "already running");
            printf("%s", data_to_send);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
            s_queue_line_status(1, client_fd, listener, fdmax, masterptr, writemasterptr);
        }
        else if (!strncasecmp("CONNECT ", client_string, 8)) {
            // new connection required to isp
            // look for the trailing space to avoid "connect" or "connectxxx"
            // Set the isp, starting at the space after the connect - it will be stripped
            set_current_isp(client_string + 7);
            // try to connect. Ok if already a child
            if (-1 == (s_queue_do_connect(1, client_fd, listener, fdmax,
                                          masterptr, writemasterptr))) {
                // snprintf(data_to_send, CLIENTBUFFSIZE, "%s\r\n",
                //          "7908 ***** Connection failed *****");
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", INFOMSG2,
                         "Warning:", "***** Connection failed *****");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
                // set_client_unconnected_both(client_fd); // done in s_queue_do_connect()
            }
            else {
                char tempbuff[30]; // For formatting
                sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                         MYCONNECTREQ, tempbuff, "connection requested");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
                set_client_connected(client_fd);
                set_client_unreconnected(client_fd);
                // tell everyone the status
                s_queue_line_status(0, -1, listener, fdmax, masterptr, writemasterptr);
                // s_queue_line_status(1, client_fd, listener, fdmax, masterptr, writemasterptr);
            }
        }
        else {
            // connector program not running but no isp specified
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s\r\n",
                     "7908 ***** Connection not valid *****");
            printf("%s", data_to_send);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
        }


        printf("connection requested %u\n", is_client_connected_either(client_fd));
        /* can't call status, as the one-shot client will do it as well */
    }


    // else if (!strcasecmp("RECONNECT", client_string)) {
    else if ((!strcasecmp("RECONNECT", client_string))
             || (!strncasecmp("RECONNECT ", client_string, 10))) {

        printf("reconnection requested %u\n", is_client_connected_either(client_fd));

        syslog(LOG_USER | LOG_INFO, "reconnection request from %s [%s]",
               get_client_name(client_fd), get_client_address(client_fd));

        if (Child_pid != getpid()) {
            // connector running
            /* check if already requested connection */
            if (0 != is_client_reconnected(client_fd)) {
                char tempbuff[30]; // For formatting
                sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                         MYRECONNECTALREADY, tempbuff, "reconnection already requested");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
            }
            else {
                char tempbuff[30]; // For formatting
                sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                         MYRECONNECTREQ, tempbuff, "reconnection requested");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
                set_client_unconnected(client_fd);
                set_client_reconnected(client_fd);
            }
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CONNECTORALREADY,
                     "Connector program:", "already running");
            printf("%s", data_to_send);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
            s_queue_line_status(1, client_fd, listener, fdmax, masterptr, writemasterptr);
        }
        else if (!strncasecmp("RECONNECT ", client_string, 10)) {
            // new connection required to isp
            // look for the trailing space to avoid "connect" or "connectxxx"
            // Set the isp, starting at the space after the connect - it will be stripped
            set_current_isp(client_string + 9);
            // try to connect. Ok if already a child
            if (-1 == (s_queue_do_connect(1, client_fd, listener, fdmax,
                                          masterptr, writemasterptr))) {
                // snprintf(data_to_send, CLIENTBUFFSIZE, "%s\r\n",
                //         "7909 ***** Connection failed *****");
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", INFOMSG2,
                         "Warning:", "***** Connection failed *****");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
            }
            else {
                char tempbuff[30]; // For formatting
                sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                         MYRECONNECTREQ, tempbuff, "reconnection requested");
                printf("%s", data_to_send);
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
                set_client_unconnected(client_fd);
                set_client_reconnected(client_fd);
                // tell everyone the status
                s_queue_line_status(0, -1, listener, fdmax, masterptr, writemasterptr);
                // s_queue_line_status(1, client_fd, listener, fdmax, masterptr, writemasterptr);
            }
        }
        else {
            // connector program not running but no isp specified
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s\r\n",
                     "7909 ***** Reconnection not valid *****");
            printf("%s", data_to_send);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
        }
        printf("reconnection requested %u\n", is_client_connected_either(client_fd));
        /* can't call status, as the one-shot client will do it as well */
    }

    else if (!strcasecmp("\002AUTORECONNECT\003", client_string)) {

        printf("auto reconnection requested %u\n", is_client_connected_either(client_fd));

        syslog(LOG_USER | LOG_INFO, "auto reconnection request for %s [%s]",
               get_client_name(client_fd), get_client_address(client_fd));

       {
            char tempbuff[30]; // For formatting
            sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                     AUTORECONNECT, tempbuff, "auto reconnection requested");
            printf("%s", data_to_send);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
            set_client_unconnected(client_fd);
            set_client_reconnected(client_fd);
        }

        // try to connect. Ok if already a child
        if (-1 == (s_queue_do_connect(1, client_fd, listener, fdmax,
                                          masterptr, writemasterptr))) {
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", INFOMSG2,
                     "Warning:", "***** Connection failed *****");
            printf("%s", data_to_send);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
        }
        printf("auto reconnection requested %u\n", is_client_connected_either(client_fd));
    }

    else if (!strcasecmp("DISCONNECT", client_string)) {
        printf("disconnection requested %u\n", is_client_connected_either(client_fd));

        if (0 != is_client_connected_either(client_fd)) {
            char tempbuff[30]; // For formatting
            sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                     MYDISCONNECTREQ, tempbuff, "disconnection requested");
            printf("%s", data_to_send);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
            /* update the clients connected flag after disconnecting and decrement counter */
            set_client_unconnected_both(client_fd);
            conditional_disconnect();
        }
        else {
            char tempbuff[30]; // For formatting
            sprintf(tempbuff, "%s [%d]:", "This client", client_fd);
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                     MYNOCONNECTREQ, tempbuff, "connection / reconnection not requested");
            printf("%s", data_to_send);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
        }
        printf("disconnection requested %u\n", is_client_connected_either(client_fd));
    }

    else if (!strcasecmp("BANG", client_string)) {
        // avoid sending lower case to clients
        if (!strcmp("BANG", client_string)) {
            char tempbuff[30]; // For formatting
            sprintf(tempbuff, "%s [%d]:", "Client", client_fd);
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n",
                     GLOBALDISCONNECTREQ, tempbuff, "global disconnection requested");
            printf("global disconnection requested %u\n", is_client_connected_either(client_fd));
            // send to connected and reconnected clients
            s_queue_cstring(data_to_send, 2, -1, listener, fdmax, masterptr, writemasterptr);
            s_queue_cstring(data_to_send, 3, -1, listener, fdmax, masterptr, writemasterptr);
            syslog(LOG_USER | LOG_INFO, "global disconnection request for %s [%s]",
                   get_client_name(client_fd), get_client_address(client_fd));

            set_all_clients_unconnected();
            set_all_clients_unreconnected();
            // kill background clients
            // unconnect first to stop conditional disconnect again in quit processing
            kill_donotsend_clients(listener, fdmax, masterptr, writemasterptr);
            conditional_disconnect();
            // syslog(LOG_USER | LOG_INFO, "global disconnection request for %s [%s]",
            //        get_client_name(client_fd), get_client_address(client_fd));

            printf("global disconnection requested %u\n", is_client_connected_either(client_fd));
        }
    }

    else if (!strcasecmp("QUIT", client_string)) {
        snprintf(data_to_send, CLIENTBUFFSIZE, "8016 Quit requested\n");
        printf("quit requested %u\n", is_client_connected_either(client_fd));

        if (0 != is_client_connected_either(client_fd)) {
            /* update the clients connected flag after disconnecting and decrement counter */
            set_client_unconnected_both(client_fd);
            conditional_disconnect();
        }

        // now kill the client connection
        close(client_fd); // bye!
        FD_CLR(client_fd, masterptr); // remove from master set
        FD_CLR(client_fd, writemasterptr); // remove from master set
        /* update maximum */
        if (client_fd >= fdmax) --fdmax;
        /* remove the client index */
        remove_client_data(client_fd);
    }

    else if (!strcasecmp("STATUS", client_string)) {
        if (0 == is_client_donotsend(client_fd)) {
             // only the server should initiate the info script
            s_queue_status(1, client_fd, listener, fdmax, masterptr, writemasterptr);
        }

    }


    else if (!strcasecmp("\002SERVERNAME\003", client_string)) {
        // Not required, done when client connects
        // But java client is too slow, so needed
        if (0 == is_client_donotsend(client_fd)) {
            // Send server name for Java client
            char *myname_p = getservername();
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s Server:  %s\r\n", SERVERNAMEMSG, myname_p);
            // send to  client only
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
        }
    }


    else if (!strcasecmp("ISPS", client_string)) {
        if (0 == is_client_donotsend(client_fd)) {
            MICEISP const  *miceisp_p = getisps(); // start of linked list of isp names
            // miceisp_p cannot be NULL
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s: %s\r\n", STARTOFISP,
                     miceisp_p->shortname, miceisp_p->longname);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
            while (NULL != (miceisp_p = miceisp_p->next)) {
                snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s: %s\r\n", ISPDATA,
                         miceisp_p->shortname, miceisp_p->longname);
                // send to  client only
                s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                                masterptr, writemasterptr);
            }
 
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s -----\r\n", ENDOFISP);
            s_queue_cstring(data_to_send, 1, client_fd, listener, fdmax,
                            masterptr, writemasterptr);
        }
    }


    else if (!strcasecmp("\002SENDISPS\003", client_string)) {
        // used by the server itself when conf file has been reread for isps.
        // called from in the select loop
        MICEISP const  *miceisp_p = getisps(); // start of linked list of isp names
        // miceisp_p cannot be NULL
        snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s: %s\r\n", STARTOFISP,
                 miceisp_p->shortname, miceisp_p->longname);
        s_queue_cstring(data_to_send, 0, -1, listener, fdmax, masterptr, writemasterptr);
        while (NULL != (miceisp_p = miceisp_p->next)) {
            snprintf(data_to_send, CLIENTBUFFSIZE, "%s %s: %s\r\n", ISPDATA,
                     miceisp_p->shortname, miceisp_p->longname);
            // send to all clients
            s_queue_cstring(data_to_send, 0, -1, listener, fdmax, masterptr, writemasterptr);
        }

        snprintf(data_to_send, CLIENTBUFFSIZE, "%s -----\r\n", ENDOFISP);
        s_queue_cstring(data_to_send, 0, -1, listener, fdmax, masterptr, writemasterptr);

    }


    else if (!strcasecmp("\002DONOTSEND\003", client_string)) {
        // don't send data to this client
        set_client_donotsend(client_fd);
    }


    else if (!strncasecmp("\002OWNERID\003", client_string, 9)) {
        // owner from background message client
        char *ptr, *msg_ptr;
        msg_ptr = client_string + 9;
        // remove multi-lines if there is an odd \r or \n
        if (NULL != (ptr = strchr(msg_ptr, '\r'))) *ptr = '\0';
        if (NULL != (ptr = strchr(msg_ptr, '\n'))) *ptr = '\0';
        set_client_owner(client_fd, msg_ptr);
    }


    else if (!strncasecmp("\002CLIENTID\003", client_string, 10)) {
        // client id from background message client
        char *ptr, *msg_ptr;
        msg_ptr = client_string + 10;
        // remove multi-lines if there is an odd \r or \n
        if (NULL != (ptr = strchr(msg_ptr, '\r'))) *ptr = '\0';
        if (NULL != (ptr = strchr(msg_ptr, '\n'))) *ptr = '\0';
        set_client_id(client_fd, msg_ptr);
    }


    else if (!strncasecmp("\002KILLCLIENT\003", client_string, 12)) {
        // client id from background message client
        int fd_k; // fd of client to kill;
        char *ptr, *msg_ptr;
        msg_ptr = client_string + 12;
        // remove multi-lines if there is an odd \r or \n
        if (NULL != (ptr = strchr(msg_ptr, '\r'))) *ptr = '\0';
        if (NULL != (ptr = strchr(msg_ptr, '\n'))) *ptr = '\0';
        if (0 < (fd_k = kill_another_client(client_fd, msg_ptr))) {
            if (NULL != get_client_data(fd_k)) {
                check_and_do_command("quit", fd_k, listener,
                                     fdmax, masterptr, writemasterptr);
            }
        }
    }


    else if (!strncasecmp("\002SERVERID\003", client_string, 10)) {
        // server id from a client
        char *msg_ptr;
        msg_ptr = client_string + 10;
        set_server_id_flag(client_fd, msg_ptr); // returns non zero if OK
        printf("Client's server id = %d\n", is_server_id_ok(client_fd));
    }


    else {
        printf("%s is not a key word\n", client_string);
        iscommand = 0;
    }
    return iscommand;
}


void s_queue_line_status(int client_flag, int client_fd, int listener, int fdmax,
                             fd_set *masterptr, fd_set *writemasterptr)
{
    char data_to_send[CLIENTBUFFSIZE]; /* buffer to send data to clients */
    MICEISP *current_isp_p = get_current_isp();
    // current isp
    snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", CURRENTISPSHORT2,
             "Current ISP:", current_isp_p->shortname);
    s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                    fdmax, masterptr, writemasterptr);
    snprintf(data_to_send, CLIENTBUFFSIZE, "%-*s %s\r\n", 25, CURRENTISPLONG2,
             current_isp_p->longname);
    s_queue_cstring(data_to_send, client_flag, client_fd, listener,
                    fdmax, masterptr, writemasterptr);
    // Line status
    // don't check if the online flag has changed as this is called when a new
    // connector program is run and the status is sent to all clients
    if (1 == get_online_flag()) {
        snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", LINEON2,
                 "Line status:", "online");
        s_queue_cstring(data_to_send, client_flag, client_fd, listener, fdmax,
                        masterptr, writemasterptr);
    }
    else if (-1 == get_online_flag()) {
        snprintf(data_to_send, CLIENTBUFFSIZE, "%s %-20s %s\r\n", LINEOFF2,
                 "Line status:", "offline");
        s_queue_cstring(data_to_send, client_flag, client_fd, listener, fdmax,
                        masterptr, writemasterptr);
    }
}


char *getnow_s(void)
{
    static char now_s[30];
    time_t now_time_t;
    struct tm *now_tm_p;
    // *now_s = '\0';
    if (-1 == time(&now_time_t)) *now_s = '\0';
    else {
        now_tm_p = localtime(&now_time_t);
        if (0 == strftime(now_s, 30, "%H:%M:%S", now_tm_p)) *now_s = '\0';
    }
    return now_s;
}




CLIENT_DATA * get_client_data(int fd) {
    /* returns pointer to data.  Could be NULL */
    CLIENT_DATA * client_data_ptr = NULL;
    client_data_ptr = client_data_set(0, fd);
    return client_data_ptr;
}

CLIENT_DATA * create_client_data(int fd) {
    /* returns pointer to data.  Could be NULL.  Could return existing data */
    CLIENT_DATA * client_data_ptr = NULL;
    client_data_ptr = client_data_set(1, fd);
    return client_data_ptr;
}

CLIENT_DATA * remove_client_data(int fd) {
    /* should return NULL. */
    CLIENT_DATA * client_data_ptr = NULL;
    client_data_ptr = client_data_set(2, fd);
    printf("client ptr for dead client = %p\n", client_data_ptr);
    return client_data_ptr;
}

int is_client_connected(int fd) {
    /* returns 0 if not, else 1 */
    int rc = 0;
    CLIENT_DATA * client_data_ptr;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        rc = client_data_ptr->connected.flag.connected;
    }
    return rc;
}

int is_client_reconnected(int fd) {
    /* returns 0 if not, else 1 */
    int rc = 0;
    CLIENT_DATA * client_data_ptr;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        rc = client_data_ptr->connected.flag.reconnected;
    }
    return rc;
}

int is_client_connected_either(int fd) {
    /* returns 0 if not, else non-zero if either */
    int rc = 0;
    CLIENT_DATA * client_data_ptr;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        rc = client_data_ptr->connected.both;
    }
    return rc;
}

int is_client_donotsend(int fd) {
    /* returns 0 if not, else non-zero if donotsend */
    int rc = 0;
    CLIENT_DATA * client_data_ptr;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        rc = client_data_ptr->donotsend;
    }
    return rc;
}

void set_client_connected(int fd) {
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        if (0 == client_data_ptr->connected.flag.connected) {
            client_data_ptr->connected.flag.connected = 1;
            ++Connect_requests;
        }
        if (0 != client_data_ptr->connected.flag.reconnected) {
            client_data_ptr->connected.flag.reconnected = 0;
            --Reconnect_requests;
        }
    }
}

void set_client_reconnected(int fd) {
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        if (0 == client_data_ptr->connected.flag.reconnected) {
            client_data_ptr->connected.flag.reconnected = 1;
            ++Reconnect_requests;
        }
        if (0 != client_data_ptr->connected.flag.connected) {
            client_data_ptr->connected.flag.connected = 0;
            --Connect_requests;
        }
    }
}

void set_client_connected_both(int fd) {
    /* sets both connected and reconnected flags */
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        if (0 == client_data_ptr->connected.flag.reconnected) {
            client_data_ptr->connected.flag.reconnected = 1;
            ++Reconnect_requests;
        }
        if (0 == client_data_ptr->connected.flag.connected) {
            client_data_ptr->connected.flag.connected = 1;
            ++Connect_requests;
        }
    }
}

void set_client_unconnected(int fd) {
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        if (0 != client_data_ptr->connected.flag.connected) {
            client_data_ptr->connected.flag.connected = 0;
            --Connect_requests;
        }
    }
}

void set_client_unreconnected(int fd) {
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        if (0 != client_data_ptr->connected.flag.reconnected) {
            client_data_ptr->connected.flag.reconnected = 0;
            --Reconnect_requests;
        }
    }
}

void set_client_unconnected_both(int fd) {
    /* resets both connected and reconnected flags */
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        if (0 != client_data_ptr->connected.flag.connected) {
            client_data_ptr->connected.flag.connected = 0;
            --Connect_requests;
        }
        if (0 != client_data_ptr->connected.flag.reconnected) {
            client_data_ptr->connected.flag.reconnected = 0;
            --Reconnect_requests;
        }
    }
}

void set_client_donotsend(int fd)
{
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        client_data_ptr->donotsend = 1; /* don't send data to this client */
    }
}

void set_client_owner(int fd, char *owner_p)
// set the owner (probably euid) of the client on the remote machine
{
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {

        strncpy(client_data_ptr->remoteowner, owner_p, NUMBERBUFFSIZE - 1);
        *(client_data_ptr->remoteowner + (NUMBERBUFFSIZE - 1)) = '\0';
        printf("remote owner  <%s>\n", client_data_ptr->remoteowner);

    }
}



void set_client_id(int fd, char *id_p)
// set the id (probably pid) of the client on the remote machine
{
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {

        strncpy(client_data_ptr->remoteid, id_p, NUMBERBUFFSIZE - 1);
        *(client_data_ptr->remoteid + (NUMBERBUFFSIZE - 1)) = '\0';
        printf("remote id  <%s>\n", client_data_ptr->remoteid);

    }
}

int kill_another_client(int calling_fd, char *tokill_id_p)
{
    // kill another client if owned by same owner as calling client
    // returns fd of client to be killed or -1
    int rc = -1;
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set2(6, calling_fd, tokill_id_p))) {
        rc = client_data_ptr->fd;
        printf("Client to be killed %s\n", client_data_ptr->remoteid);
    }
    return rc;
}



int set_server_id_flag(int fd, char * test_id)
{
    // returns 0 if not, else non-zero if OK
    int rc = 0;
    CLIENT_DATA * client_data_ptr = NULL;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        if (0 == strcmp(test_id, get_magic_id())) {
            client_data_ptr->has_id = 1; // 1 for Id OK
            rc = 1;
        }
        else client_data_ptr->has_id = 0;
    }
    return rc;
}


int is_server_id_ok(int fd)
{
    // returns 0 if not, else non-zero if OK
    int rc = 0;
    CLIENT_DATA * client_data_ptr;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        rc = (int)client_data_ptr->has_id;
    }
    return rc;
}


int get_ayt_status(int fd, int *client_type, int *client_count)
{
    // this includes the client_data_ptr check
    // returns 0 if off or client data not found, +1 if checking is on, -1 if timed out
    int rc = 0;
    int ayt_threshold;
    CLIENT_DATA * client_data_ptr;
    int count = 0;
    int type = CLIENTUNKNOWN;
    ayt_threshold = getayt();
    if (0 == ayt_threshold) {
        // off
        rc = 0;
        type = CLIENTUNKNOWN;
        count = 0;
    }
    // threshold not 0
    else if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        rc = 1;
        type = client_data_ptr->client_type;
        count = client_data_ptr->ayt_count;
            // now check for time out
        if (count > ayt_threshold) {
            rc = -1;  // assume dead
        }
    }
    else {
        rc = 0;
        type = CLIENTUNKNOWN;
        count = 0;
    }
    *client_type = type;
    *client_count = count;
    return rc;
}


void increment_ayt_count(int fd)
{
    CLIENT_DATA * client_data_ptr;
    if (0 != getayt()) {  // not off
        if (NULL != (client_data_ptr = client_data_set(0, fd))) {
            // client data found
            if (CLIENTUNKNOWN != client_data_ptr->client_type) {
                // Don't increment unknown clients
                // Old clients will be marked as unknown
                if (INT_MAX > client_data_ptr->ayt_count) client_data_ptr->ayt_count += 1;
            }
        }
    }
}

int reset_ayt_count(int fd, int type)
{
    // called when response from client
    // return old count, or 0 if not retrieved
    CLIENT_DATA * client_data_ptr;
    int old_count = 0;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        old_count = client_data_ptr->ayt_count;
        client_data_ptr->ayt_count = 0;   // remove this line for testing
        client_data_ptr->client_type = type;
    }
    return old_count;
}

char *get_client_name(int fd)
{
    char *rcp = "unknown";
    CLIENT_DATA * client_data_ptr;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        rcp = client_data_ptr->nameptr;
    }
    return rcp;
}

char *get_client_address(int fd)
{
    char *rcp = "unknown";
    CLIENT_DATA * client_data_ptr;
    if (NULL != (client_data_ptr = client_data_set(0, fd))) {
        rcp = client_data_ptr->ip_addressptr;
    }
    return rcp;
}


int get_client_count(void)
{
    int *count_ptr;
    count_ptr = client_data_set(5, -1);
    return *count_ptr;
}


void set_all_clients_unconnected(void)
{
    // When the child dies, the zombie handler calls this routine to
    // reset the connected flags to 0 in all the client structures.
    // Also used for global disconnection */
    client_data_set(3, -1);
    Connect_requests = 0;
}

void set_all_clients_unreconnected(void)
{
    client_data_set(4, -1);
    Reconnect_requests = 0;
}

void * client_data_set(int action, int fd)
{
    // a bit of a kludge to allow another parameter to be added to the function call elsewhere
    return client_data_set2(action, fd, NULL);
}


void * client_data_set2(int action, int fd, char *data_p)
{
    /* action 0 to identify only */
    /* action 1 to add or identify */
    // action 2 to remove, returns client_data_ptr which should be NULL
    // action 3 set all clients unconnected, returns NULL
    // action 4 set all clients unreconnected, returns NULL
    // action 5 client count, returns int *
    // action 6 The client fd is attempting to kill the client with id in data_p.
    //          Return client_data_ptr if allowed or NULL
    static int first_time = 1;
    static int maxindex = 0;   // to limit the loops
    static int client_count;  // static because address is returned.
    static CLIENT_DATA * client_data[FD_SETSIZE];  // array of pointers
    int ctr;
    CLIENT_DATA * client_data_ptr = NULL;
    void *rptr = NULL;  // this is the return value
    if (first_time) {
        /* initialise */
        printf("fd_setsize = %d\n", (int) FD_SETSIZE);
        first_time = 0;
        for (ctr = 0; ctr < FD_SETSIZE; ctr++) {
            client_data[ctr] = NULL;
        }
    }

    if (action == 0) {
        // identify only
        for (ctr = 0; (ctr < FD_SETSIZE) && (ctr <= maxindex); ctr++) {
            if ((client_data[ctr] != NULL) && (client_data[ctr]->fd == fd)) {
                // identify
                client_data_ptr = client_data[ctr];
                break;
            }
        }
        rptr = client_data_ptr;
    }

    else if (action == 1) {
        // add or identify
        for (ctr = 0; (ctr < FD_SETSIZE) && (ctr <= maxindex); ctr++) {
            if ((client_data[ctr] != NULL) && (client_data[ctr]->fd == fd)) {
                // identify
                client_data_ptr = client_data[ctr];
                break;
            }
        }
        // not found if client_data_ptr still == NULL and we are setting
        if (NULL == client_data_ptr) {
            // find the first empty index
            for (ctr = 0; ctr < FD_SETSIZE; ctr++) {
                if (NULL == client_data[ctr]) {
                    // try to create a client data structure
                    if (NULL != (client_data_ptr = client_data[ctr] = alloc_client_data())) {
                        if (ctr > maxindex) maxindex = ctr;
                        client_data_ptr->lineptr = client_data_ptr->buffptr = client_data_ptr->buff;
                        client_data_ptr->fd = fd;
                        client_data_ptr->connected.both = 0;
                        client_data_ptr->buffsize = sizeof(client_data_ptr->buff);
                        *(client_data_ptr->ip_addressptr) = '\0';
                        strcpy(client_data_ptr->nameptr, "Machine without a name");
                        more_client_data(fd, client_data_ptr);
                        // send buffer
                        *(client_data_ptr->sendbuff) = '\0'; // not neccessary
                        // decent size allocated now reallocs have been tested
                        // important that these are initialized to 0
                        client_data_ptr->bytes_sent = client_data_ptr->bytes_notsent = 0;
                        client_data_ptr->donotsend = 0;
                        client_data_ptr->has_id = 0; // not yet a script client
                        *(client_data_ptr->remoteowner) = '\0';  // remote euid
                        *(client_data_ptr->remoteid) = '\0';  // remote pid
                        client_data_ptr->client_type = CLIENTUNKNOWN;  // unknown
                        client_data_ptr->ayt_count = 0;  // none yet
                    }
                    break;
                }
            }
        }
        rptr = client_data_ptr;
    }

    /*
    else if (action == 1) {
        // add or identify
        for (ctr = 0; (ctr < FD_SETSIZE) && (ctr <= maxindex); ctr++) {
            if ((client_data[ctr] != NULL) && (client_data[ctr]->fd == fd)) {
                // identify
                client_data_ptr = client_data[ctr];
                break;
            }
        }
        // not found if client_data_ptr still == NULL and we are setting
        if (NULL == client_data_ptr) {
            // find the first empty index
            for (ctr = 0; ctr < FD_SETSIZE; ctr++) {
                if (NULL == client_data[ctr]) {
                    if (ctr > maxindex) maxindex = ctr;
                    // create a client data structure
                    client_data_ptr = client_data[ctr] = alloc_client_data();
                    client_data_ptr->lineptr = client_data_ptr->buffptr = client_data_ptr->buff;
                    client_data_ptr->fd = fd;
                    client_data_ptr->connected.both = 0;
                    client_data_ptr->buffsize = sizeof(client_data_ptr->buff);
                    *(client_data_ptr->ip_addressptr) = '\0';
                    strcpy(client_data_ptr->nameptr, "Machine without a name");
                    more_client_data(fd, client_data_ptr);
                    // send buffer
                    *(client_data_ptr->sendbuff) = '\0'; // not neccessary
                    // decent size allocated now reallocs have been tested
                    // important that these are initialized to 0
                    client_data_ptr->bytes_sent = client_data_ptr->bytes_notsent = 0;
                    client_data_ptr->donotsend = 0;
                    client_data_ptr->has_id = 0; // not yet a script client
                    *(client_data_ptr->remoteowner) = '\0';  // remote euid
                    *(client_data_ptr->remoteid) = '\0';  // remote pid
                    client_data_ptr->client_type = CLIENTUNKNOWN;  // unknown
                    client_data_ptr->ayt_count = 0;  // none yet
                    break;
                }
            }
        }
        rptr = client_data_ptr;
    }
    */

    else if (action == 2) {
        // remove
        // note:  maxindex cannot be reduced
        for (ctr = 0; ctr < FD_SETSIZE; ctr++) {
            if ((NULL != client_data[ctr]) && client_data[ctr]->fd == fd) {
                if (0 != client_data[ctr]->connected.flag.connected) {
                    client_data[ctr]->connected.flag.connected = 0; // this not neccessary
                    --Connect_requests;
                }
                if (0 != client_data[ctr]->connected.flag.reconnected) {
                    client_data[ctr]->connected.flag.reconnected = 0; // this not neccessary
                    --Reconnect_requests;
                }
                client_data_ptr = client_data[ctr] = free_client_data(client_data[ctr]);
                break;
            }
        }
        rptr = client_data_ptr;
    }

    else if (action == 3) {
        /* set all clients unconnected */
        for (ctr = 0; (ctr < FD_SETSIZE) && (ctr <= maxindex); ctr++) {
            if (NULL != client_data[ctr]) {
                /* got one */
                client_data[ctr]->connected.flag.connected = 0;
            }
        }
        rptr = NULL;
    }
    else if (action == 4) {
        /* set all clients unreconnected */
        for (ctr = 0; (ctr < FD_SETSIZE) && (ctr <= maxindex); ctr++) {
            if (NULL != client_data[ctr]) {
                /* got one */
                client_data[ctr]->connected.flag.reconnected = 0;
            }
        }
        rptr = NULL;
    }

    else if (action == 5) {
        // client count
        for (ctr = 0, client_count = 0; (ctr < FD_SETSIZE) && (ctr <= maxindex); ctr++) {
            if (client_data[ctr] != NULL)  {
                // count
                ++client_count;
            }
        }
        rptr = &client_count;
    }

    else if (action == 6) {
        // The client fd is attempting to kill the client with id in data_p.
        // Return client_data_ptr * fd if allowed or NULL.
        CLIENT_DATA * calling_client_data_ptr = NULL;
        // identify calling client
        for (ctr = 0; (ctr < FD_SETSIZE) && (ctr <= maxindex); ctr++) {
            if ((client_data[ctr] != NULL) && (client_data[ctr]->fd == fd)) {
                // identify
                calling_client_data_ptr = client_data[ctr];
                break;
            }
        }
        // If calling client OK, look for another client with the specified id that
        // has the same ip address and owner as the calling calling client.
        for (ctr = 0; (ctr < FD_SETSIZE) && (ctr <= maxindex); ctr++) {
            if ((client_data[ctr] != NULL) && (client_data[ctr]->fd != fd)) {
                // compare
                if (('\0' != *(client_data[ctr]->remoteid))
                    && (!strcmp(data_p, client_data[ctr]->remoteid))
                    && ('\0' != *(client_data[ctr]->ip_addressptr))
                    && ('\0' != *(client_data[ctr]->remoteowner))
                    && (!strcmp(calling_client_data_ptr->remoteowner,
                                client_data[ctr]->remoteowner))
                    && (!strcmp(calling_client_data_ptr->ip_addressptr,
                                client_data[ctr]->ip_addressptr))) {
                    // found
                    rptr = client_data[ctr];
                }
            }
        }
    }

    return rptr;  // could be NULL
}



CLIENT_DATA * alloc_client_data(void)
// could return NULL
{
    CLIENT_DATA * new_client_data_ptr;
    char * name_ptr, * ip_address_ptr, * sendbuff;
    new_client_data_ptr = malloc(sizeof(CLIENT_DATA));
    name_ptr = malloc(30 * sizeof(char)); // to be realloced later
    ip_address_ptr = malloc(sizeof(char)); // to be realloced later
    sendbuff = malloc(SENDBUFFSIZE * sizeof(char));

    // debug
    // ip_address_ptr = NULL; // for testing

    if ((NULL == new_client_data_ptr)
        || (NULL == name_ptr)
        || (NULL == ip_address_ptr)
        || (NULL == sendbuff)) {
        printf("%s\n", "malloc error in alloc_client_data");
        // error stuff needed
        free(sendbuff); // OK if NULL
        free(ip_address_ptr);  // how do we detect error?
        free(name_ptr);  // how do we detect error?
        free(new_client_data_ptr);  /* how do we detect error? */
        new_client_data_ptr = NULL;
    }
    else {
        printf("client data structure created %p\n", new_client_data_ptr);
        printf("sizeof buff %d\n", sizeof(new_client_data_ptr->buff));
        new_client_data_ptr->buffsize = sizeof(new_client_data_ptr->buff);
        new_client_data_ptr->nameptr = name_ptr;
        new_client_data_ptr->ip_addressptr = ip_address_ptr;
        new_client_data_ptr->sendbuff = sendbuff;
        new_client_data_ptr->sendbuffsize = SENDBUFFSIZE;
    }
    return new_client_data_ptr;
}

/*
CLIENT_DATA * alloc_client_data(void) {
    CLIENT_DATA * new_client_data_ptr;
    char * name_ptr, * ip_address_ptr, * sendbuff;
    new_client_data_ptr = malloc(sizeof(CLIENT_DATA));
    name_ptr = malloc(30 * sizeof(char)); // to be realloced later
    ip_address_ptr = malloc(sizeof(char)); // to be realloced later
    
    sendbuff = malloc(SENDBUFFSIZE * sizeof(char));
    if ((NULL == new_client_data_ptr)
        || (NULL == name_ptr)
        || (NULL == ip_address_ptr)
        || (NULL == sendbuff))
        perror("malloc client data structure");
    // error stuff needed
    else {
        printf("client data structure created %p\n", new_client_data_ptr);
        printf("sizeof buff %d\n", sizeof(new_client_data_ptr->buff));
        new_client_data_ptr->buffsize = sizeof(new_client_data_ptr->buff);
        new_client_data_ptr->nameptr = name_ptr;
        new_client_data_ptr->ip_addressptr = ip_address_ptr;
        new_client_data_ptr->sendbuff = sendbuff;
        new_client_data_ptr->sendbuffsize = SENDBUFFSIZE;
    }
    return new_client_data_ptr;
}
*/


void more_client_data(int fd, CLIENT_DATA *data_ptr) {
    struct sockaddr_in sockstuff;
    socklen_t sockstufflen;
    struct hostent *hoststuff;
    int len;
    int sin_addr_len;
    char *ptr;
    /* first the ip address */
    sockstufflen = sizeof(struct sockaddr_in);
    if (-1 == getpeername(fd, &sockstuff, &sockstufflen)) {
        perror("getpeername");
    }

    else {
        len = strlen(inet_ntoa(sockstuff.sin_addr));
        if (NULL != (ptr = realloc(data_ptr->ip_addressptr, len + 1))) {
            sin_addr_len = sizeof(sockstuff.sin_addr);
            /* error checking here */
            data_ptr->ip_addressptr = ptr;
            strcpy(ptr, inet_ntoa(sockstuff.sin_addr));
            printf("ip address from sockstuff: %s\n", data_ptr->ip_addressptr);


            // printf("sin_addr length %d\n", sin_addr_len);
            // sleep(4);


            /* now get the name */
            // if (NULL == (hoststuff = gethostbyaddr((char *)&sockstuff.sin_addr, 4, AF_INET))) {
            if (NULL == (hoststuff = gethostbyaddr((char *)&sockstuff.sin_addr, sin_addr_len, AF_INET))) {
                herror("gethostbyaddr");
                /* exit(1); */
            }
            else {
                printf("%s\n", "got gethostbyaddr");
                len = strlen(hoststuff->h_name);
                if (NULL != (ptr = realloc(data_ptr->nameptr, len + 1))) {
                    /* error checking here */
                    data_ptr->nameptr = ptr;
                    strcpy(ptr, hoststuff->h_name);
                    printf("name from sockstuff: %s\n", data_ptr->nameptr);
                }
            }
        }
    }
}


CLIENT_DATA * free_client_data(CLIENT_DATA *client_data_ptr) {
    printf("freeing client data structure %p\n", client_data_ptr);
    free(client_data_ptr->sendbuff); /* OK if NULL */
    free(client_data_ptr->ip_addressptr);  /* how do we detect error? */
    free(client_data_ptr->nameptr);  /* how do we detect error? */
    free(client_data_ptr);  /* how do we detect error? */
    client_data_ptr = NULL;
    return client_data_ptr;  /* hopefully NULL */
}



void auto_reconnect(int listener, int fdmax, fd_set *masterptr, fd_set *writemasterptr)
{
    int fd_j;
    CLIENT_DATA * client_data_ptr = NULL;
    for(fd_j = 0; fd_j <= fdmax; fd_j++) {
        // everyone who wants reconnection,  except the listener
        if ((FD_ISSET(fd_j, masterptr)) && (fd_j != listener)) {
            if (0 != is_client_reconnected(fd_j)) {
                if (NULL != (client_data_ptr = get_client_data(fd_j))) {
                    // check_and_do_command("\002autoreconnect\003", client_data_ptr, fd_j, listener,
                    //                     fdmax, masterptr, writemasterptr);
                    check_and_do_command("\002autoreconnect\003", fd_j, listener,
                                         fdmax, masterptr, writemasterptr);
                }
            }
        }
    }
}


int is_reconnect_required(int listener, int fdmax, fd_set *masterptr)
{
    int fd_ctr, reconnected = 0;
    for(fd_ctr = 0; fd_ctr <= fdmax; fd_ctr++) {
        if ((FD_ISSET(fd_ctr, masterptr)) && (fd_ctr != listener)) {
            if (0 != is_client_reconnected(fd_ctr)) {
                reconnected = 1;
            }
        }
    }
    return reconnected;  // 0 if no reconnections wanted
}


void kill_donotsend_clients(int listener, int fdmax, fd_set *masterptr, fd_set *writemasterptr)
{
    int fd_j;
    CLIENT_DATA * client_data_ptr = NULL;
    for(fd_j = 0; fd_j <= fdmax; fd_j++) {
        // everyone who is donotsend,  except the listener
        if ((FD_ISSET(fd_j, masterptr)) && (fd_j != listener)) {
            if (0 != is_client_donotsend(fd_j)) {
                if (NULL != (client_data_ptr = get_client_data(fd_j))) {
                    check_and_do_command("quit", fd_j, listener,
                                         fdmax, masterptr, writemasterptr);
                }
            }
        }
    }
}


char *get_message(void)
{
   return message_data_set(NULL, 2);
}

void set_message(char *message)
{
    message_data_set(message, 1);
}

int cmp_message(void)
{
    // returns 0 if the two message buffers are identical
    return strcmp(message_data_set(NULL, 2), message_data_set(NULL, 3));
}

char * message_data_set(char *message, int action)
{
    // action 1: set message
    // action 2: get message
    // action 3: get previous message
    static char cbuff[2][CLIENTBUFFSIZE + 1] = {"", ""};
    static int cbuff_ctr = 0;
    static char *cbuff_ptr = cbuff[0];

    if (1 == action) {
        //switch buffers
        cbuff_ctr = (cbuff_ctr + 1) % 2;
        cbuff_ptr = cbuff[cbuff_ctr];
        // only if the connector child is running
        if ((NULL == message) || (Child_pid == getpid())) {
            *cbuff[0] = '\0';
            *cbuff[1] = '\0';
        }
        else {
            strncpy(cbuff_ptr, message, CLIENTBUFFSIZE);
            *(cbuff_ptr + CLIENTBUFFSIZE) = '\0';
        }
    }
    else if (2 == action) {
        // return current cbuff
        cbuff_ptr = cbuff[cbuff_ctr];
    }
    else if (3 == action) {
        // return previous buffer
        cbuff_ptr = cbuff[(cbuff_ctr + 1) % 2];
    }
    else *cbuff_ptr = '\0';

    return cbuff_ptr;
}




char *get_magic_id(void)
{
    // This id is used by the client program to identify itself when sending the
    // online status.  This is to prevent user clients from sending dummy online
    // messages.
    static char magic_id[NUMBERBUFFSIZE] = "";  // make sure it is big enough
    static char magic_id_env[NUMBERBUFFSIZE + 10] = "MICEID=";  // make sure it is big enough
    unsigned int seed;
    int magic_num;
    if ('\0' == *magic_id) {
        // first time
        seed = (unsigned int)getpid();  // pid_t is what?
        srand(seed);
        magic_num = rand();
        while (0 == magic_num) magic_num = rand();  // 0 is used for testing in the client
        // convert to a cstring
        snprintf(magic_id, NUMBERBUFFSIZE, "%d", magic_num);
        strcat(magic_id_env, magic_id);
        // check for demo server
        if (!strcmp(DEMOLOGIDENT, Params2p->logident)) {
            printf("Server id = %d\n", magic_num);
        }
        // add the magic number to the environment for use in the scripts
        if (0 != putenv(magic_id_env)) {
            // log if error
            syslog(LOG_USER | LOG_INFO, "%s", "Could not add environment variable");
        }
    }
    return magic_id;
}



int get_online_flag(void)
{
   return online_flag_set(0, 2);
}

void set_online_flag(int flag)
{
    online_flag_set(flag, 1);
}

int cmp_online_flag(void)
{
    // returns 0 if the two flags are identical
    if (online_flag_set(0, 2) == online_flag_set(0, 3)) return 0;
    else return 1;
}

int online_flag_set(int flag, int action)
{
    // action 1: set flag
    // action 2: get flag
    // action 3: get previous flag
    static int flagbuff[2] = {0, 0};
    static int flagbuff_ctr = 0;
    static int *flagbuff_ptr = &flagbuff[0];


    if (1 == action) {
        //switch buffers
        flagbuff_ctr = (flagbuff_ctr + 1) % 2;
        flagbuff_ptr = &flagbuff[flagbuff_ctr];
        // only if the connector child is running
        if ((0 == flag) || (Child_pid == getpid())) {
            flagbuff[0] = 0;
            flagbuff[1] = 0;
        }
        else {
            *flagbuff_ptr = flag;
        }
    }
    else if (2 == action) {
        // return current cbuff
        flagbuff_ptr = &flagbuff[flagbuff_ctr];
    }
    else if (3 == action) {
        // return previous buffer
        flagbuff_ptr = &flagbuff[(flagbuff_ctr + 1) % 2];
    }
    else *flagbuff_ptr = 0;

    return *flagbuff_ptr;
}



int get_options(int argc, char *argv[], STARTUP_ARGS *startup)
{
    // char * port_p = NULL;
    // int optind_temp, arg_error = 0;
    int arg_error = 0;
    char *ptr;
    // int exit_flag = 0;
    // set exit_flag to 1 for help and 2 for version and exit if server type is not inetd.

    struct option long_options[] = {
        {"help", no_argument, 0, 'h'},
        {"demo", no_argument, 0, 'D'},
        {"port", required_argument, 0, 'p'},
        {"file", required_argument, 0, 'f'},
        {"version", no_argument, 0, 'V'},
        {"verbose", no_argument, 0, 'v'},
        {"inetd", no_argument, 0, 'I'},
        {0,0,0,0}
    };
    int optchar;

    // startup->port_usi = 0;
    startup->server_type = SERVER_UNDEFINED; // not daemon so exit_flag can be tested
    startup->server_mode = SERVER_UNDEFINED;  // see defines, mice or demo
    startup->port_p = NULL;
    startup->port_hs = 0;
    startup->conffile_p = NULL;
    startup->V_flag = 0;

    while (1) {
        optchar = getopt_long(argc, argv, "hDvVIp:f:", long_options, NULL);
        if (EOF == optchar) break;
        switch (optchar) {
        case 'h' :  // printf("%s %s\n", "option h", optarg);
            do_help();
            // exit_flag = 1;
            exit(0);
            break;
        case 'D' :  // printf("%s %s\n", "option D", optarg);
            // paramsupdate(SERVER_DEMO, NULL, NULL); // Set to demo server.
            startup->server_mode = SERVER_DEMO;
            // setparams(1);  // Set to demo server.
            break;
        case 'V' :  // printf("%s %s\n", "option v", optarg);
            startup->V_flag = 1;
            printf("\n\n%s\n%s\n%s\n\n", Mice_server_version, Copyright, Disclaimer);
            // exit_flag = 2;
            // exit(0);
            break;
        case 'v' :  // printf("%s %s\n", "option v", optarg);
            startup->server_type = SERVER_STANDALONE; // terminal server;
            break;
        case 'I' :  // printf("%s %s\n", "option I", optarg);
            if (SERVER_STANDALONE != startup->server_type) {
                // -v has precedence
                startup->server_type = SERVER_INETD; // inetd server;
            }
            break;
        case 'p' : // printf("%s %s\n", "option p", optarg);
            startup->port_p = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case 'f' : // printf("%s %s\n", "option f", optarg);
            startup->conffile_p = optarg;
            // printf("datafile: %s\n", startup->conffile_p);
            break;
        case '?' : // printf("%s\n", "unknown option character");
            ++arg_error;
            break;
        case ':' : // printf("%s\n", "missing parameter");
            ++arg_error;
            break;
        default : fprintf(stderr, "what's this %c\n", optchar);
        ++arg_error;
        break;
        }
    }


    // port.  This has precedence over the -p option
    if ((argc - optind) > 0) {
        startup->port_p = argv[optind + 0]; // ready for more options
    }


    if (NULL != (ptr = (argc - optind) > 1 ? "too many" :
                 (argc - optind) < 0 ? "not enough" : NULL)) {  // ready for more options
        int optind_temp = optind;
        fprintf(stderr, "%s non-options %d\n", ptr, (argc - optind));
        while (optind_temp < argc) {
            fprintf(stderr, "non-option is %s\n", argv[optind_temp++]);
        }
        ++arg_error;
    }

    // check port
    if (NULL != startup->port_p) {
        if (0 == (startup->port_hs = check_port(startup->port_p))) {
            fprintf(stderr, "Invalid port:  %s\n", startup->port_p);
            ++arg_error;
        }
    }

    // check conf file
    if (NULL != startup->conffile_p) {
        // check this file exists
        struct stat statbuff;
        if (0 != stat(startup->conffile_p, &statbuff)) {
            perror(startup->conffile_p);
            ++arg_error;
        }
        else if (!(S_ISREG(statbuff.st_mode))) {
            // and is a regular file
            fprintf(stderr, "%s: not a regular file\n", startup->conffile_p);
            ++arg_error;
        }
        else if (0 != access(startup->conffile_p, R_OK)) {
            // and is readable
            perror(startup->conffile_p);
            ++arg_error;
        }
    }


    if (arg_error) {
        fprintf(stderr, "argument errors = %d\n", arg_error);
        exit(1);
    }
    // At this point, server can be standalone or undefined.
    // Exit of not inetd server and exit_flag has been set
    // if ((0 != exit_flag) && (SERVER_INETD != startup->server_type)) {
    //    exit(0);
    // }

    return arg_error;
}


unsigned short int check_port(char *port_p)
{
    // return numeric port number or 0 if error
    unsigned short int port = 0; // default
    struct servent *miceent;
    unsigned long port_ul;
    char *endptr;
    port_ul = strtoul(port_p, &endptr, 10);
    // printf("port_ul %lu\n", port_ul);
    if (('\0' == *endptr) && (USHRT_MAX >= port_ul) && (0 < port_ul))   // valid number
        port = (unsigned short int)port_ul;
    else if (NULL != (miceent = getservbyname(port_p, "tcp")))
        port = ntohs(miceent->s_port);
    // back to host short syntax is ntohs(miceent->s_port)
    return port;
}




void do_help(void)
{
    char *prog = "mice.server";

    printf("\nusage:  %s [port] [option]\n\n", prog);
    printf("%s\n\t%s\n", "-p #, --port #", "Port number");
    printf("-I, --inetd\n\t%s\n", "inetd server");
    printf("-v, --verbose\n\t%s\n", "verbose server");
    printf("-D, --demo\n\t%s\n","demonstration mode");
    printf("-f filename, --file filename\n\tConiguration file\n");
    printf("-V, --version\n\t%s version and parameters\n", prog);
    printf("%s\n\t%s\n", "-h, --help", "Help information");
    printf("\nexample:  %s\n", prog);
    printf("example:  %s --demo\n", prog);
    printf("example:  %s mice\n", prog);
    printf("example:  %s -f /usr/local/etc/anothermice.conf\n", prog);
    printf("example:  %s -p 5022\n\n", prog);
    // if (SERVER_STANDALONE == server_type) exit(1);
}



int create_socket(STARTUP_ARGS *startup)
{
    int listener;     // listening socket descriptor
    struct sockaddr_in myaddr;     // server address
    // get the listener
    if (-1 == (listener = socket(PF_INET, SOCK_STREAM, 0))) {
        perror("socket");
        exit(1);
    }

    {
        int flag;
        // flag = fcntl(listener, F_GETFD);
        // printf("fcntl on listener %d\n", flag);
        flag = fcntl(listener, F_SETFD, 1L); /* close on exec flag */
        // flag = fcntl(listener, F_GETFD);
        // printf("fcntl on listener %d\n", flag);

        // set non-blocking
        if (-1 == (fcntl(listener, F_SETFL, O_NONBLOCK))) { // non-blocking
            perror("listener non-block");
        }
        // flag = fcntl(listener, F_GETFL);


    }

    {
        // lose "address already in use" error message
        /* get rid of this? see man 2 bind  */
        int yes = 1;
        if (-1 == setsockopt(listener, SOL_SOCKET, SO_REUSEADDR, &yes,
                       sizeof(int))) {
            perror("setsockopt");
            // exit(1);
        }
    }


    {
        struct linger sock_linger;
        int linger_size = sizeof(struct linger);
        if (-1 == getsockopt(listener, SOL_SOCKET, SO_LINGER,
                             &sock_linger, &linger_size)) {
            perror("getsockopt");
            // exit(1);
        }
        else {
            printf("linger %d:%d\n", sock_linger.l_onoff, sock_linger.l_linger);

            if((0 == sock_linger.l_onoff) || (500 > sock_linger.l_linger)) {
                sock_linger.l_onoff = 1;
                sock_linger.l_linger = 500;
                if (-1 == setsockopt(listener, SOL_SOCKET, SO_LINGER,
                                     &sock_linger, linger_size)) {
                    perror("setsockopt");
                    // exit(1);
                }
                else {
                    printf("linger %d:%d\n", sock_linger.l_onoff, sock_linger.l_linger);
                }
            }
        }
    }


    // bind
    myaddr.sin_family = AF_INET;
    myaddr.sin_addr.s_addr = INADDR_ANY;
    myaddr.sin_port = htons( Params2p->port);  // short, network byte order
    memset(&(myaddr.sin_zero), '\0', 8);
    if (bind(listener, (struct sockaddr *)&myaddr, sizeof(myaddr)) == -1) {
        perror("bind");
        exit(1);
    }

    // listen
    if (listen(listener, 10) == -1) {
        perror("listen");
        exit(1);
    }
    return listener;
}

char *getservername(void)
{
    // my name
    static char myname[MYNAMESIZE] = "";
    if ('\0' == *myname) {
        // first time
        int rc;
        rc = gethostname(myname, MYNAMESIZE);
        if ((-1 == rc) || (MYNAMESIZE == rc)) {
            perror("gethostname");
            strcpy(myname, "mice.server");
        }
    }
    return myname;
}


